import models from '../models/index.js';
import { Op } from 'sequelize';
import { generateTokenPair, generateRandomToken, verifyToken } from '../utils/tokens.js';
import { sendEmail, emailTemplates } from '../utils/email.js';
import logger from '../utils/logger.js';

// Register a new user
export const register = async (req, res, next) => {
  try {
    const { name, email, password } = req.body;

    // Check if user already exists
    const existingUser = await models.User.findOne({ where: { email } });
    if (existingUser) {
      return res.status(400).json({
        status: 'error',
        message: 'User with this email already exists'
      });
    }

    // Generate verification token
    const verificationToken = generateRandomToken();

    // Create new user
    const user = await models.User.create({
      name,
      email,
      password,
      verificationToken,
      role: 'user', // Default role
      status: 'active',
      emailVerified: false
    });

    // Generate verification URL
    const verificationUrl = `${process.env.FRONTEND_URL}/verify-email/${verificationToken}`;

    // Send verification email
    await sendEmail({
      to: email,
      ...emailTemplates.welcome(name, verificationUrl)
    });

    // Generate tokens
    const { accessToken, refreshToken } = generateTokenPair(user.id);

    // Store refresh token
    await models.Session.create({
      userId: user.id,
      refreshToken,
      userAgent: req.headers['user-agent'],
      ipAddress: req.ip,
      expiresAt: new Date(Date.now() + 7 * 24 * 60 * 60 * 1000) // 7 days
    });

    // Return user data and tokens
    return res.status(201).json({
      status: 'success',
      message: 'User registered successfully. Please check your email to verify your account.',
      data: {
        user: {
          id: user.id,
          name: user.name,
          email: user.email,
          role: user.role,
          emailVerified: user.emailVerified
        },
        tokens: {
          accessToken,
          refreshToken
        }
      }
    });
  } catch (error) {
    next(error);
  }
};

// Register an admin user
export const registerAdmin = async (req, res, next) => {
  try {
    const { name, email, password, adminSecret } = req.body;

    // Verify admin secret
    if (adminSecret !== process.env.ADMIN_SECRET) {
      return res.status(403).json({
        status: 'error',
        message: 'Invalid admin secret'
      });
    }

    // Check if user already exists
    const existingUser = await models.User.findOne({ where: { email } });
    if (existingUser) {
      return res.status(400).json({
        status: 'error',
        message: 'User with this email already exists'
      });
    }

    // Generate verification token
    const verificationToken = generateRandomToken();

    // Create new admin user
    const user = await models.User.create({
      name,
      email,
      password,
      verificationToken,
      role: 'admin',
      status: 'active',
      emailVerified: true // Auto-verify admin accounts
    });

    // Generate tokens
    const { accessToken, refreshToken } = generateTokenPair(user.id);

    // Store refresh token
    await models.Session.create({
      userId: user.id,
      refreshToken,
      userAgent: req.headers['user-agent'],
      ipAddress: req.ip,
      expiresAt: new Date(Date.now() + 7 * 24 * 60 * 60 * 1000) // 7 days
    });

    // Return user data and tokens
    return res.status(201).json({
      status: 'success',
      message: 'Admin user registered successfully',
      data: {
        user: {
          id: user.id,
          name: user.name,
          email: user.email,
          role: user.role,
          emailVerified: user.emailVerified
        },
        tokens: {
          accessToken,
          refreshToken
        }
      }
    });
  } catch (error) {
    next(error);
  }
};

// Register a membership user
export const registerMembership = async (req, res, next) => {
  try {
    const { name, email, password, memberSecret } = req.body;

    // Verify member secret
    if (memberSecret !== process.env.ADMIN_SECRET) {
      return res.status(403).json({
        status: 'error',
        message: 'Invalid member secret'
      });
    }

    // Check if user already exists
    const existingUser = await models.User.findOne({ where: { email } });
    if (existingUser) {
      return res.status(400).json({
        status: 'error',
        message: 'User with this email already exists'
      });
    }

    // Generate verification token
    const verificationToken = generateRandomToken();

    // Create new membership user
    const user = await models.User.create({
      name,
      email,
      password,
      verificationToken,
      role: 'membershipUser',
      status: 'active',
      emailVerified: true // Auto-verify membership accounts
    });

    // Generate tokens
    const { accessToken, refreshToken } = generateTokenPair(user.id);

    // Store refresh token
    await models.Session.create({
      userId: user.id,
      refreshToken,
      userAgent: req.headers['user-agent'],
      ipAddress: req.ip,
      expiresAt: new Date(Date.now() + 7 * 24 * 60 * 60 * 1000) // 7 days
    });

    // Return user data and tokens
    return res.status(201).json({
      status: 'success',
      message: 'Membership user registered successfully',
      data: {
        user: {
          id: user.id,
          name: user.name,
          email: user.email,
          role: user.role,
          emailVerified: user.emailVerified
        },
        tokens: {
          accessToken,
          refreshToken
        }
      }
    });
  } catch (error) {
    next(error);
  }
};

// Login user
export const login = async (req, res, next) => {
  try {
    const { email, password } = req.body;

    // Find user
    const user = await models.User.findOne({ where: { email } });
    if (!user) {
      return res.status(401).json({
        status: 'error',
        message: 'Invalid credentials'
      });
    }

    // Check if user is banned
    if (user.status === 'banned') {
      return res.status(403).json({
        status: 'error',
        message: 'Your account has been banned. Please contact support for more information.'
      });
    }

    // Verify password
    const isPasswordValid = await user.comparePassword(password);
    if (!isPasswordValid) {
      return res.status(401).json({
        status: 'error',
        message: 'Invalid credentials'
      });
    }

    // Generate tokens
    const { accessToken, refreshToken } = generateTokenPair(user.id);

    // Store refresh token
    await models.Session.create({
      userId: user.id,
      refreshToken,
      userAgent: req.headers['user-agent'],
      ipAddress: req.ip,
      expiresAt: new Date(Date.now() + 7 * 24 * 60 * 60 * 1000) // 7 days
    });

    // Update last login
    user.lastLogin = new Date();
    await user.save();

    // Return user data and tokens
    return res.status(200).json({
      status: 'success',
      message: 'Login successful',
      data: {
        user: {
          id: user.id,
          name: user.name,
          email: user.email,
          role: user.role,
          emailVerified: user.emailVerified
        },
        tokens: {
          accessToken,
          refreshToken
        }
      }
    });
  } catch (error) {
    next(error);
  }
};

// Refresh token
export const refreshToken = async (req, res, next) => {
  try {
    const { refreshToken } = req.body;

    // Find session with this refresh token
    const session = await models.Session.findOne({
      where: {
        refreshToken,
        isRevoked: false,
        expiresAt: {
          [Op.gt]: new Date()
        }
      },
      include: {
        model: models.User,
        as: 'user'
      }
    });

    if (!session) {
      return res.status(401).json({
        status: 'error',
        message: 'Invalid or expired refresh token'
      });
    }

    // Check if user is banned
    if (session.user.status === 'banned') {
      // Revoke all sessions for this user
      await models.Session.update(
        { isRevoked: true },
        { where: { userId: session.userId } }
      );

      return res.status(403).json({
        status: 'error',
        message: 'Your account has been banned. Please contact support for more information.'
      });
    }

    // Generate new tokens
    const { accessToken, refreshToken: newRefreshToken } = generateTokenPair(session.userId);

    // Revoke old refresh token
    session.isRevoked = true;
    await session.save();

    // Store new refresh token
    await models.Session.create({
      userId: session.userId,
      refreshToken: newRefreshToken,
      userAgent: req.headers['user-agent'],
      ipAddress: req.ip,
      expiresAt: new Date(Date.now() + 7 * 24 * 60 * 60 * 1000) // 7 days
    });

    // Return new tokens
    return res.status(200).json({
      status: 'success',
      message: 'Token refreshed successfully',
      data: {
        tokens: {
          accessToken,
          refreshToken: newRefreshToken
        }
      }
    });
  } catch (error) {
    next(error);
  }
};

// Logout user
export const logout = async (req, res, next) => {
  try {
    const authHeader = req.headers.authorization;
    if (!authHeader || !authHeader.startsWith('Bearer ')) {
      return res.status(400).json({
        status: 'error',
        message: 'No token provided'
      });
    }

    const refreshToken = req.body.refreshToken;
    if (!refreshToken) {
      return res.status(400).json({
        status: 'error',
        message: 'Refresh token is required'
      });
    }

    // Revoke refresh token
    await models.Session.update(
      { isRevoked: true },
      { 
        where: { 
          refreshToken,
          userId: req.user.id
        } 
      }
    );

    return res.status(200).json({
      status: 'success',
      message: 'Logout successful'
    });
  } catch (error) {
    next(error);
  }
};

// Verify email
export const verifyEmail = async (req, res, next) => {
  try {
    const { token } = req.params;

    // Find user with this verification token
    const user = await models.User.findOne({
      where: { verificationToken: token }
    });

    if (!user) {
      return res.status(400).json({
        status: 'error',
        message: 'Invalid or expired verification token'
      });
    }

    // Update user
    user.emailVerified = true;
    user.verificationToken = null;
    await user.save();

    return res.status(200).json({
      status: 'success',
      message: 'Email verified successfully'
    });
  } catch (error) {
    next(error);
  }
};

// Forgot password
export const forgotPassword = async (req, res, next) => {
  try {
    const { email } = req.body;

    // Find user with this email
    const user = await models.User.findOne({ where: { email } });
    if (!user) {
      // Don't reveal that the user doesn't exist
      return res.status(200).json({
        status: 'success',
        message: 'If your email is registered, you will receive a password reset link'
      });
    }

    // Generate reset token
    const resetToken = generateRandomToken();
    
    // Set token and expiry
    user.resetPasswordToken = resetToken;
    user.resetPasswordExpires = new Date(Date.now() + 3600000); // 1 hour
    await user.save();

    // Generate reset URL
    const resetUrl = `${process.env.FRONTEND_URL}/reset-password/${resetToken}`;

    // Send password reset email
    await sendEmail({
      to: email,
      ...emailTemplates.passwordReset(user.name, resetUrl)
    });

    return res.status(200).json({
      status: 'success',
      message: 'If your email is registered, you will receive a password reset link'
    });
  } catch (error) {
    next(error);
  }
};

// Reset password
export const resetPassword = async (req, res, next) => {
  try {
    const { token } = req.params;
    const { password } = req.body;

    // Find user with this reset token and valid expiry
    const user = await models.User.findOne({
      where: {
        resetPasswordToken: token,
        resetPasswordExpires: {
          [Op.gt]: new Date()
        }
      }
    });

    if (!user) {
      return res.status(400).json({
        status: 'error',
        message: 'Invalid or expired reset token'
      });
    }

    // Update password
    user.password = password;
    user.resetPasswordToken = null;
    user.resetPasswordExpires = null;
    await user.save();

    // Revoke all existing sessions for this user
    await models.Session.update(
      { isRevoked: true },
      { where: { userId: user.id } }
    );

    return res.status(200).json({
      status: 'success',
      message: 'Password reset successful. Please login with your new password.'
    });
  } catch (error) {
    next(error);
  }
};