import models from '../models/index.js';
import logger from '../utils/logger.js';
import { sequelize } from '../models/index.js';
import { Op } from 'sequelize';



// Create a new challenge
export const createChallenge = async (req, res, next) => {
  try {
    const { ctfId, title, description, category, points, difficulty, flag, hints, files, videoLink, image } = req.body;
    const creatorId = req.user.id;

    if (!['admin', 'membershipUser'].includes(req.user.role)) {
      return res.status(403).json({
        status: 'error',
        message: 'You do not have permission to create challenges'
      });
    }

    const ctf = await models.CTF.findByPk(ctfId);
    if (!ctf) {
      return res.status(404).json({ status: 'error', message: 'CTF not found' });
    }

    if (ctf.creatorId !== creatorId && req.user.role !== 'admin') {
      return res.status(403).json({ status: 'error', message: 'You do not have permission to add challenges to this CTF' });
    }

    const maxOrder = await models.Challenge.max('order', { where: { ctfId } });

    const challenge = await models.Challenge.create({
      ctfId,
      title,
      description,
      category,
      points,
      difficulty,
      flag,
      hints: hints || [],
      files: files || [],
      videoLink: videoLink || null,
      image: image || null,
      order: (maxOrder || 0) + 1,
      isActive: true
    });

    if (ctf.status === 'approved' && req.user.role !== 'admin') {
      ctf.status = 'pending';
      await ctf.save();

      const admins = await models.User.findAll({ where: { role: 'admin' } });
      for (const admin of admins) {
        await models.Notification.create({
          userId: admin.id,
          type: 'ctf_update',
          message: `CTF "${ctf.title}" has been updated with new challenges and needs approval.`,
          read: false,
          actionUrl: `/admin/ctfs/pending`
        });
      }
    }

    return res.status(201).json({
      status: 'success',
      message: 'Challenge created successfully',
      data: {
        id: challenge.id,
        title: challenge.title,
        ctfId: challenge.ctfId
      }
    });
  } catch (error) {
    next(error);
  }
};

// Get challenge by ID
export const getChallengeById = async (req, res, next) => {
  try {
    const { id } = req.params;

    const challenge = await models.Challenge.findByPk(id, {
      include: [
        {
          model: models.CTF,
          as: 'ctf',
          include: [
            {
              model: models.User,
              as: 'creator',
              attributes: ['id', 'name']
            }
          ]
        }
      ]
    });

    if (!challenge) {
      return res.status(404).json({
        status: 'error',
        message: 'Challenge not found'
      });
    }

    const isCtfActive = challenge.ctf.status === 'approved' &&
      new Date() >= new Date(challenge.ctf.startDate) &&
      new Date() <= new Date(challenge.ctf.endDate);

    const isCreatorOrAdmin =
      req.user.id === challenge.ctf.creatorId ||
      req.user.role === 'admin';

    if (!isCtfActive && !isCreatorOrAdmin) {
      return res.status(403).json({
        status: 'error',
        message: 'You do not have permission to view this challenge'
      });
    }

    const solved = await models.Submission.findOne({
      where: {
        userId: req.user.id,
        challengeId: id,
        correct: true
      }
    });

    const solveCount = await models.Submission.count({
      where: {
        challengeId: id,
        correct: true
      }
    });

    const formattedChallenge = {
      id: challenge.id,
      title: challenge.title,
      description: challenge.description,
      category: challenge.category,
      points: challenge.points,
      difficulty: challenge.difficulty,
      hints: challenge.hints,
      files: challenge.files,
      videoLink: challenge.videoLink,
      image: challenge.image,
      ctf: {
        id: challenge.ctf.id,
        title: challenge.ctf.title,
        creator: challenge.ctf.creator ? {
          id: challenge.ctf.creator.id,
          name: challenge.ctf.creator.name
        } : null
      },
      isSolved: !!solved,
      solveCount,
      ...(isCreatorOrAdmin && { flag: challenge.flag })
    };

    return res.status(200).json({
      status: 'success',
      data: formattedChallenge
    });
  } catch (error) {
    next(error);
  }
};

// Update a challenge
export const updateChallenge = async (req, res, next) => {
  try {
    const { id } = req.params;
    const { title, description, category, points, difficulty, flag, hints, files, videoLink, image } = req.body;

    const challenge = await models.Challenge.findByPk(id, {
      include: [{ model: models.CTF, as: 'ctf' }]
    });

    if (!challenge) {
      return res.status(404).json({
        status: 'error',
        message: 'Challenge not found'
      });
    }

    const isCreatorOrAdmin =
      req.user.id === challenge.ctf.creatorId ||
      req.user.role === 'admin';

    if (!isCreatorOrAdmin) {
      return res.status(403).json({
        status: 'error',
        message: 'You do not have permission to update this challenge'
      });
    }

    if (title) challenge.title = title;
    if (description) challenge.description = description;
    if (category) challenge.category = category;
    if (points) challenge.points = points;
    if (difficulty) challenge.difficulty = difficulty;
    if (flag) challenge.flag = flag;
    if (hints) challenge.hints = hints;
    if (files) challenge.files = files;
    if (videoLink !== undefined) challenge.videoLink = videoLink;
    if (image !== undefined) challenge.image = image;

    await challenge.save();

    if (challenge.ctf.status === 'approved' && req.user.role !== 'admin') {
      challenge.ctf.status = 'pending';
      await challenge.ctf.save();

      const admins = await models.User.findAll({
        where: {
          role: 'admin'
        }
      });

      for (const admin of admins) {
        await models.Notification.create({
          userId: admin.id,
          type: 'ctf_update',
          message: `CTF "${challenge.ctf.title}" has been updated with modified challenges and needs approval.`,
          read: false,
          actionUrl: `/admin/ctfs/pending`
        });
      }
    }

    return res.status(200).json({
      status: 'success',
      message: 'Challenge updated successfully',
      data: {
        id: challenge.id,
        title: challenge.title
      }
    });
  } catch (error) {
    next(error);
  }
};
// Delete a challenge
export const deleteChallenge = async (req, res, next) => {
  let transaction;
  
  try {
    transaction = await sequelize.transaction();
    
    const { id } = req.params;
    
    // Get challenge with its CTF
    const challenge = await models.Challenge.findByPk(id, {
      include: [
        {
          model: models.CTF,
          as: 'ctf',
          attributes: ['id', 'creatorId', 'status']
        }
      ],
      transaction
    });
    
    if (!challenge) {
      await transaction.rollback();
      return res.status(404).json({
        status: 'error',
        message: 'Challenge not found'
      });
    }
    
    // Check if user can delete this challenge
    const canDelete = 
      req.user.role === 'admin' || 
      (req.user.role === 'membershipUser' && challenge.ctf.creatorId === req.user.id);
    
    if (!canDelete) {
      await transaction.rollback();
      return res.status(403).json({
        status: 'error',
        message: 'You do not have permission to delete this challenge'
      });
    }
    
    // Delete all submissions for this challenge
    await models.Submission.destroy({
      where: {
        challengeId: id
      },
      transaction
    });
    
    // Delete the challenge
    await challenge.destroy({ transaction });
    
    // If CTF was approved and user is not admin, set back to pending
    if (challenge.ctf.status === 'approved' && req.user.role !== 'admin') {
      await models.CTF.update(
        { status: 'pending' },
        { 
          where: { id: challenge.ctfId },
          transaction
        }
      );
      
      // Notify admins about the change
      const admins = await models.User.findAll({
        where: {
          role: 'admin'
        }
      });
      
      for (const admin of admins) {
        await models.Notification.create({
          userId: admin.id,
          type: 'ctf_update',
          message: `CTF has been updated (challenge deleted) and needs review.`,
          read: false,
          actionUrl: `/admin/ctfs/pending`,
          transaction
        });
      }
    }
    
    // Commit transaction
    await transaction.commit();
    
    return res.status(200).json({
      status: 'success',
      message: 'Challenge deleted successfully'
    });
  } catch (error) {
    // Rollback transaction if it exists
    if (transaction) await transaction.rollback();
    
    // Log the error
    logger.error('Error deleting challenge:', {
      error: error.message,
      stack: error.stack,
      userId: req.user?.id,
      challengeId: req.params.id
    });

    next(error);
  }
};

// Submit a flag for a challenge
export const submitFlag = async (req, res, next) => {
  try {
    const { id } = req.params;
    const { flag } = req.body;
    const userId = req.user.id;
    
    // Get challenge
    const challenge = await models.Challenge.findByPk(id, {
      include: [
        {
          model: models.CTF,
          as: 'ctf'
        }
      ]
    });
    
    if (!challenge) {
      return res.status(404).json({
        status: 'error',
        message: 'Challenge not found'
      });
    }
    
    // Check if CTF is active
    const now = new Date();
    const isActive = 
      challenge.ctf.status === 'approved' && 
      now >= new Date(challenge.ctf.startDate) && 
      now <= new Date(challenge.ctf.endDate);
    
    // if (!isActive) {
    //   return res.status(403).json({
    //     status: 'error',
    //     message: 'This CTF is not currently active'
    //   });
    // }
    
    // Check if user already solved this challenge
    const existingSolve = await models.Submission.findOne({
      where: {
        userId,
        challengeId: id,
        correct: true
      }
    });
    
    if (existingSolve) {
      return res.status(400).json({
        status: 'error',
        message: 'You have already solved this challenge'
      });
    }
    
    // Count previous attempts
    const attemptCount = await models.Submission.count({
      where: {
        userId,
        challengeId: id
      }
    });
    
    // Check if maximum attempts reached
    if (challenge.maxAttempts && attemptCount >= challenge.maxAttempts) {
      return res.status(400).json({
        status: 'error',
        message: `Maximum attempts (${challenge.maxAttempts}) reached for this challenge`
      });
    }
    
    // Check if flag is correct
    const isCorrect = flag === challenge.flag;
    
    // Calculate points
    let pointsAwarded = 0;
    if (isCorrect) {
      // If dynamic scoring is enabled, calculate points based on solve count
      if (challenge.dynamicScoring) {
        const solveCount = await models.Submission.count({
          where: {
            challengeId: id,
            correct: true
          }
        });
        
        // Example dynamic scoring formula (decreases with more solves)
        pointsAwarded = Math.max(
          Math.floor(challenge.points * Math.pow(0.95, solveCount)),
          Math.floor(challenge.points * 0.5)
        );
      } else {
        pointsAwarded = challenge.points;
      }
      
      // Update challenge solve count
      challenge.solveCount += 1;
      await challenge.save();
    }
    
    // Create submission record
    const submission = await models.Submission.create({
      userId,
      challengeId: id,
      flag,
      correct: isCorrect,
      pointsAwarded: isCorrect ? pointsAwarded : 0,
      attemptNumber: attemptCount + 1,
      ipAddress: req.ip,
      userAgent: req.headers['user-agent'],
      solveTime: isCorrect ? Math.floor((now - new Date(challenge.ctf.startDate)) / 1000) : null
    });
    
    // If correct, create notification
    if (isCorrect) {
      await models.Notification.create({
        userId,
        type: 'challenge_solved',
        message: `You solved "${challenge.title}" and earned ${pointsAwarded} points!`,
        read: false,
        actionUrl: `/ctfs/${challenge.ctfId}`
      });
      
      // Check if first blood (first to solve this challenge)
      if (challenge.solveCount === 1) {
        // Award an achievement
        await models.Achievement.create({
          userId,
          title: 'First Blood',
          description: `First to solve "${challenge.title}"`,
          category: 'solving',
          points: Math.floor(challenge.points * 0.1) // Bonus points for first blood
        });
      }
      
      // Check for category-based achievements
      await checkCategoryAchievements(userId, challenge.category);
    }
    
    return res.status(200).json({
      status: 'success',
      data: {
        correct: isCorrect,
        pointsAwarded: isCorrect ? pointsAwarded : 0,
        message: isCorrect 
          ? `Congratulations! You solved the challenge and earned ${pointsAwarded} points.`
          : 'Incorrect flag. Try again.'
      }
    });
  } catch (error) {
    next(error);
  }
};

// Helper function to check and award category-based achievements
const checkCategoryAchievements = async (userId, category) => {
  try {
    // Count solved challenges in this category
    const solvedCount = await models.Submission.count({
      where: {
        userId,
        correct: true
      },
      include: [
        {
          model: models.Challenge,
          as: 'challenge',
          where: {
            category
          },
          required: true
        }
      ]
    });
    
    // Define achievement thresholds
    const achievementThresholds = {
      3: {
        title: `${category} Beginner`,
        description: `Solved 3 challenges in the ${category} category`,
        points: 10
      },
      5: {
        title: `${category} Intermediate`,
        description: `Solved 5 challenges in the ${category} category`,
        points: 25
      },
      10: {
        title: `${category} Expert`,
        description: `Solved 10 challenges in the ${category} category`,
        points: 50
      },
      20: {
        title: `${category} Master`,
        description: `Solved 20 challenges in the ${category} category`,
        points: 100
      }
    };
    
    // Check if user reached any achievement threshold
    if (achievementThresholds[solvedCount]) {
      const achievement = achievementThresholds[solvedCount];
      
      // Check if user already has this achievement
      const existingAchievement = await models.Achievement.findOne({
        where: {
          userId,
          title: achievement.title
        }
      });
      
      if (!existingAchievement) {
        // Award the achievement
        await models.Achievement.create({
          userId,
          title: achievement.title,
          description: achievement.description,
          category: 'category',
          points: achievement.points
        });
        
        // Create notification
        await models.Notification.create({
          userId,
          type: 'achievement',
          message: `You earned the "${achievement.title}" achievement!`,
          read: false
        });
      }
    }
  } catch (error) {
    logger.error('Error checking category achievements:', error);
  }
};

// Get all challenges
export const getAllChallenges = async (req, res, next) => {
  try {
    const {
      page = 1,
      limit = 10,
      category,
      difficulty,
      search,
      ctfId,
      sortBy = 'createdAt',
      sortOrder = 'DESC'
    } = req.query;

    // Build where clause
    const where = {};

    // Add CTF filter if provided
    if (ctfId) {
      where.ctfId = ctfId;

      // Get CTF to check permissions
      const ctf = await models.CTF.findByPk(ctfId);
      if (!ctf) {
        return res.status(404).json({
          status: 'error',
          message: 'CTF not found'
        });
      }

      // Check if user can view challenges
      const isCtfActive =
        ctf.status === 'approved' &&
        new Date() >= new Date(ctf.startDate) &&
        new Date() <= new Date(ctf.endDate);

      const isCreatorOrAdmin =
        req.user.id === ctf.creatorId ||
        req.user.role === 'admin';

      if (!isCtfActive && !isCreatorOrAdmin) {
        return res.status(403).json({
          status: 'error',
          message: 'You do not have permission to view these challenges'
        });
      }
    }

    // Add category filter
    if (category) {
      where.category = category;
    }

    // Add difficulty filter
    if (difficulty) {
      where.difficulty = difficulty;
    }

    // Add search filter
    if (search) {
      where[Op.or] = [
        { title: { [Op.iLike]: `%${search}%` } },
        { description: { [Op.iLike]: `%${search}%` } },
        { category: { [Op.iLike]: `%${search}%` } }
      ];
    }

    // Calculate offset
    const offset = (page - 1) * limit;

    // Get challenges with pagination and include CTF + creator
    const { count, rows: challenges } = await models.Challenge.findAndCountAll({
      where,
      include: [
        {
          model: models.CTF,
          as: 'ctf',
          attributes: ['id', 'title', 'status', 'startDate', 'endDate', 'creatorId'],
          include: [
            {
              model: models.User,
              as: 'creator',
              attributes: ['id', 'name']
            }
          ]
        }
      ],
      order: [[sortBy, sortOrder]],
      limit: parseInt(limit),
      offset: offset
    });

    // Get solved challenges for current user
    const solvedChallenges = await models.Submission.findAll({
      where: {
        userId: req.user.id,
        correct: true,
        challengeId: {
          [Op.in]: challenges.map(c => c.id)
        }
      },
      attributes: ['challengeId']
    });

    const solvedChallengeIds = new Set(solvedChallenges.map(s => s.challengeId));

    // Format challenges with safe checks
    const formattedChallenges = challenges.map(challenge => {
      if (!challenge.ctf) {
        // Handle challenges without a CTF gracefully
        return {
          id: challenge.id,
          title: challenge.title,
          description: challenge.description,
          category: challenge.category,
          points: challenge.points,
          difficulty: challenge.difficulty,
          solveCount: challenge.solveCount,
          isSolved: solvedChallengeIds.has(challenge.id),
          ctf: null,
          isActive: false
        };
      }

      const isCtfActive =
        challenge.ctf.status === 'approved' &&
        new Date() >= new Date(challenge.ctf.startDate) &&
        new Date() <= new Date(challenge.ctf.endDate);

      const isCreatorOrAdmin =
        req.user.id === challenge.ctf.creator?.id ||
        req.user.role === 'admin';

      return {
        id: challenge.id,
        title: challenge.title,
        description: challenge.description,
        category: challenge.category,
        points: challenge.points,
        difficulty: challenge.difficulty,
        solveCount: challenge.solveCount,
        isSolved: solvedChallengeIds.has(challenge.id),
        ctf: {
          id: challenge.ctf.id,
          title: challenge.ctf.title,
          status: challenge.ctf.status,
          startDate: challenge.ctf.startDate,
          endDate: challenge.ctf.endDate,
          creator: challenge.ctf.creator ? {
            id: challenge.ctf.creator.id,
            name: challenge.ctf.creator.name
          } : null
        },
        ...(isCreatorOrAdmin && {
          flag: challenge.flag,
          hints: challenge.hints,
          files: challenge.files
        }),
        isActive: isCtfActive
      };
    });

    // Get unique categories for filtering
    const categories = await models.Challenge.findAll({
      attributes: [[sequelize.fn('DISTINCT', sequelize.col('category')), 'category']],
      where: {
        category: {
          [Op.not]: null
        }
      },
      raw: true
    });

    // Get unique difficulties for filtering
    const difficulties = await models.Challenge.findAll({
      attributes: [[sequelize.fn('DISTINCT', sequelize.col('difficulty')), 'difficulty']],
      where: {
        difficulty: {
          [Op.not]: null
        }
      },
      raw: true
    });

    return res.status(200).json({
      status: 'success',
      data: {
        challenges: formattedChallenges,
        pagination: {
          total: count,
          page: parseInt(page),
          totalPages: Math.ceil(count / limit),
          limit: parseInt(limit)
        },
        filters: {
          categories: categories.map(c => c.category),
          difficulties: difficulties.map(d => d.difficulty)
        }
      }
    });
  } catch (error) {
    logger.error('Error getting challenges:', {
      error: error.message,
      stack: error.stack,
      userId: req.user?.id
    });
    next(error);
  }
};