import models from '../models/index.js';
import logger from '../utils/logger.js';

// Create a new custom CTF page
export const createCustomCTFPage = async (req, res, next) => {
  try {
    const { title, html, css, assets, ctfId } = req.body;
    const creatorId = req.user.id;

    // Create the custom CTF page
    const customPage = await models.CustomCTFPage.create({
      title,
      html,
      css,
      assets: assets || [],
      creatorId,
      ctfId,
      status: 'published'
    });

    return res.status(201).json({
      status: 'success',
      message: 'Custom CTF page created successfully',
      data: {
        id: customPage.id,
        title: customPage.title,
        status: customPage.status
      }
    });
  } catch (error) {
    next(error);
  }
};

// Get custom CTF page by ID
export const getCustomCTFPage = async (req, res, next) => {
  try {
    const { id } = req.params;

    const page = await models.CustomCTFPage.findByPk(id, {
      include: [
        {
          model: models.User,
          as: 'creator',
          attributes: ['id', 'name']
        },
        {
          model: models.CTF,
          as: 'ctf',
          attributes: ['id', 'title']
        }
      ]
    });

    if (!page) {
      return res.status(404).json({
        status: 'error',
        message: 'Custom CTF page not found'
      });
    }

    return res.status(200).json({
      status: 'success',
      data: page
    });
  } catch (error) {
    next(error);
  }
};

// Update custom CTF page
export const updateCustomCTFPage = async (req, res, next) => {
  try {
    const { id } = req.params;
    const { title, html, css, assets, ctfId } = req.body;

    const page = await models.CustomCTFPage.findByPk(id);

    if (!page) {
      return res.status(404).json({
        status: 'error',
        message: 'Custom CTF page not found'
      });
    }

    // Check if user has permission to update
    if (page.creatorId !== req.user.id && req.user.role !== 'admin') {
      return res.status(403).json({
        status: 'error',
        message: 'You do not have permission to update this page'
      });
    }

    // Update the page
    await page.update({
      title: title || page.title,
      html: html || page.html,
      css: css || page.css,
      assets: assets || page.assets,
      ctfId: ctfId || page.ctfId
    });

    return res.status(200).json({
      status: 'success',
      message: 'Custom CTF page updated successfully',
      data: {
        id: page.id,
        title: page.title
      }
    });
  } catch (error) {
    next(error);
  }
};

// Delete custom CTF page
export const deleteCustomCTFPage = async (req, res, next) => {
  try {
    const { id } = req.params;

    const page = await models.CustomCTFPage.findByPk(id);

    if (!page) {
      return res.status(404).json({
        status: 'error',
        message: 'Custom CTF page not found'
      });
    }

    // Check if user has permission to delete
    if (page.creatorId !== req.user.id && req.user.role !== 'admin') {
      return res.status(403).json({
        status: 'error',
        message: 'You do not have permission to delete this page'
      });
    }

    // Delete the page
    await page.destroy();

    return res.status(200).json({
      status: 'success',
      message: 'Custom CTF page deleted successfully'
    });
  } catch (error) {
    next(error);
  }
};

// Get all custom CTF pages for a user
export const getUserCustomCTFPages = async (req, res, next) => {
  try {
    const userId = req.user.id;
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 10;
    const offset = (page - 1) * limit;

    const { count, rows } = await models.CustomCTFPage.findAndCountAll({
      where: {
        creatorId: userId
      },
      include: [
        {
          model: models.CTF,
          as: 'ctf',
          attributes: ['id', 'title']
        }
      ],
      order: [['createdAt', 'DESC']],
      limit,
      offset
    });

    return res.status(200).json({
      status: 'success',
      data: {
        pages: rows,
        pagination: {
          total: count,
          currentPage: page,
          totalPages: Math.ceil(count / limit),
          limit
        }
      }
    });
  } catch (error) {
    next(error);
  }
};