import models from '../models/index.js';

// Get all CTFs created by the member
export const getMemberCTFs = async (req, res, next) => {
  try {
    const memberId = req.user.id;
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 10;
    const offset = (page - 1) * limit;
    
    // Get all CTFs created by the member
    const { count, rows } = await models.CTF.findAndCountAll({
      where: {
        creatorId: memberId
      },
      include: [
        {
          model: models.Challenge,
          as: 'challenges',
          attributes: ['id', 'title']
        }
      ],
      order: [['createdAt', 'DESC']],
      limit,
      offset
    });
    
    // Get participation statistics for each CTF
    const ctfsWithStats = await Promise.all(rows.map(async (ctf) => {
      const submissions = await models.Submission.findAll({
        include: [
          {
            model: models.Challenge,
            as: 'challenge',
            where: {
              ctfId: ctf.id
            },
            required: true
          }
        ],
        attributes: ['userId', 'correct']
      });
      
      const participants = new Set(submissions.map(s => s.userId));
      const totalParticipants = participants.size;
      
      const correctSubmissions = submissions.filter(s => s.correct).length;
      const totalSubmissions = submissions.length;
      const solveRate = totalSubmissions > 0 ? (correctSubmissions / totalSubmissions) * 100 : 0;
      
      return {
        ...ctf.toJSON(),
        stats: {
          totalParticipants,
          totalSubmissions,
          correctSubmissions,
          solveRate: Math.round(solveRate * 100) / 100 // Round to 2 decimal places
        }
      };
    }));
    
    return res.status(200).json({
      status: 'success',
      data: {
        ctfs: ctfsWithStats,
        pagination: {
          total: count,
          currentPage: page,
          totalPages: Math.ceil(count / limit),
          limit
        }
      }
    });
  } catch (error) {
    next(error);
  }
};

// Get member statistics
export const getMemberStats = async (req, res, next) => {
  try {
    const memberId = req.user.id;
    
    // Get all CTFs created by the member
    const ctfs = await models.CTF.findAll({
      where: {
        creatorId: memberId
      },
      include: [
        {
          model: models.Challenge,
          as: 'challenges'
        }
      ]
    });
    
    // Calculate stats
    const ctfsCreated = ctfs.length;
    
    let totalParticipants = 0;
    let totalPointsAwarded = 0;
    
    // Process each CTF for detailed statistics
    for (const ctf of ctfs) {
      const submissions = await models.Submission.findAll({
        include: [
          {
            model: models.Challenge,
            as: 'challenge',
            where: {
              ctfId: ctf.id
            },
            required: true
          }
        ],
        attributes: ['userId', 'pointsAwarded']
      });
      
      const participants = new Set(submissions.map(s => s.userId));
      totalParticipants += participants.size;
      
      totalPointsAwarded += submissions.reduce((sum, s) => sum + (s.pointsAwarded || 0), 0);
    }
    
    // Calculate average rating (placeholder - would need rating system)
    const averageRating = 4.5;
    
    // Calculate impact metrics (placeholder - would need more detailed tracking)
    const usersReached = totalParticipants;
    const learningHours = Math.round(totalParticipants * 2.5); // Assumption: average 2.5 hours per participant
    const positiveFeedback = Math.round(totalParticipants * 0.6); // Assumption: 60% positive feedback
    
    return res.status(200).json({
      status: 'success',
      data: {
        contributionStats: {
          ctfsCreated,
          totalParticipants,
          averageRating,
          totalPointsAwarded
        },
        impactMetrics: {
          usersReached,
          learningHours,
          positiveFeedback
        }
      }
    });
  } catch (error) {
    next(error);
  }
};