import { validationResult } from 'express-validator';

// Helper to format validation errors
const formatValidationErrors = (errors) => {
  const formattedErrors = {};
  
  errors.array().forEach(error => {
    const field = error.param;
    if (!formattedErrors[field]) {
      formattedErrors[field] = [];
    }
    formattedErrors[field].push(error.msg);
  });
  
  return Object.entries(formattedErrors).map(([field, messages]) => ({
    field,
    messages
  }));
};

// Middleware to validate request
export const validate = (req, res, next) => {
  const errors = validationResult(req);
  
  if (!errors.isEmpty()) {
    // Get the first error for each field
    const formattedErrors = formatValidationErrors(errors);
    
    return res.status(400).json({
      status: 'error',
      message: 'Validation failed',
      errors: formattedErrors
    });
  }
  
  // Check for required fields in nested objects
  if (req.body.challenges) {
    const challengeErrors = [];
    
    req.body.challenges.forEach((challenge, index) => {
      const missingFields = [];
      
      // Required fields for challenges
      const requiredFields = ['title', 'description', 'category', 'points', 'flag'];
      
      requiredFields.forEach(field => {
        if (!challenge[field]) {
          missingFields.push(field);
        }
      });
      
      if (missingFields.length > 0) {
        challengeErrors.push({
          index,
          field: 'challenges',
          messages: [`Challenge at index ${index} is missing required fields: ${missingFields.join(', ')}`]
        });
      }
    });
    
    if (challengeErrors.length > 0) {
      return res.status(400).json({
        status: 'error',
        message: 'Challenge validation failed',
        errors: challengeErrors
      });
    }
  }
  
  next();
};