import { Model, DataTypes } from 'sequelize';

class CTF extends Model {
  static init(sequelize) {
    return super.init(
      {
        id: {
          type: DataTypes.UUID,
          defaultValue: DataTypes.UUIDV4,
          primaryKey: true
        },
        title: {
          type: DataTypes.STRING,
          allowNull: false,
          validate: {
            notEmpty: true,
            len: [3, 200]
          }
        },
        description: {
          type: DataTypes.TEXT,
          allowNull: false
        },
        startDate: {
          type: DataTypes.DATE,
          allowNull: false,
          validate: {
            isDate: true
          }
        },
        endDate: {
          type: DataTypes.DATE,
          allowNull: false,
          validate: {
            isDate: true,
            isAfterStartDate(value) {
              if (new Date(value) <= new Date(this.startDate)) {
                throw new Error('End date must be after start date');
              }
            }
          }
        },
        creatorId: {
          type: DataTypes.UUID,
          allowNull: false,
          references: {
            model: 'users',
            key: 'id'
          }
        },
        status: {
          type: DataTypes.ENUM('draft', 'pending', 'approved', 'rejected', 'active', 'completed'),
          defaultValue: 'draft',
          allowNull: false
        },
        difficulty: {
          type: DataTypes.ENUM('beginner', 'intermediate', 'advanced', 'expert'),
          allowNull: false
        },
        visibility: {
          type: DataTypes.ENUM('public', 'private', 'invitation'),
          defaultValue: 'public',
          allowNull: false
        },
        tags: {
          type: DataTypes.JSON,
          defaultValue: []
        },
        reviewFeedback: {
          type: DataTypes.TEXT,
          allowNull: true
        },
        isActive: {
          type: DataTypes.VIRTUAL,
          get() {
            const now = new Date();
            return (
              this.status === 'approved' &&
              now >= new Date(this.startDate) &&
              now <= new Date(this.endDate)
            );
          }
        }
      },
      {
        sequelize,
        modelName: 'ctf',
        timestamps: true
      }
    );
  }

  // Define associations
  static associate(models) {
    this.belongsTo(models.User, { foreignKey: 'creatorId', as: 'creator' });
    this.hasMany(models.Challenge, { foreignKey: 'ctfId', as: 'challenges' });
  }
}

export default CTF;