import { Model, DataTypes } from 'sequelize';

class Challenge extends Model {
  static init(sequelize) {
    return super.init(
      {
        id: {
          type: DataTypes.UUID,
          defaultValue: DataTypes.UUIDV4,
          primaryKey: true
        },
        ctfId: {
          type: DataTypes.UUID,
          allowNull: false,
          references: {
            model: 'ctfs',
            key: 'id'
          }
        },
        title: {
          type: DataTypes.STRING,
          allowNull: false,
          validate: {
            notEmpty: true,
            len: [3, 200]
          }
        },
        description: {
          type: DataTypes.TEXT,
          allowNull: false
        },
        category: {
          type: DataTypes.STRING,
          allowNull: false,
          validate: {
            notEmpty: true
          }
        },
        points: {
          type: DataTypes.INTEGER,
          allowNull: false,
          validate: {
            min: 0
          }
        },
        difficulty: {
          type: DataTypes.ENUM('easy', 'medium', 'hard', 'extreme'),
          allowNull: false
        },
        flag: {
          type: DataTypes.STRING,
          allowNull: false,
          validate: {
            notEmpty: true
          }
        },
        hints: {
          type: DataTypes.JSON,
          defaultValue: []
        },
        files: {
          type: DataTypes.JSON,
          defaultValue: []
        },
        order: {
          type: DataTypes.INTEGER,
          defaultValue: 0
        },
        isActive: {
          type: DataTypes.BOOLEAN,
          defaultValue: true
        },
        maxAttempts: {
          type: DataTypes.INTEGER,
          allowNull: true
        },
        solveCount: {
          type: DataTypes.INTEGER,
          defaultValue: 0
        },
        dynamicScoring: {
          type: DataTypes.BOOLEAN,
          defaultValue: false
        },
        videoLink: {
          type: DataTypes.STRING,
          allowNull: true,
          validate: {
            isUrl: true
          }
        },
        image: {
          type: DataTypes.STRING,
          allowNull: true,
          validate: {
            isUrl: true
          }
        }
      },
      {
        sequelize,
        modelName: 'challenge',
        timestamps: true
      }
    );
  }

  static associate(models) {
    this.belongsTo(models.CTF, { foreignKey: 'ctfId', as: 'ctf' });
    this.hasMany(models.Submission, { foreignKey: 'challengeId', as: 'submissions' });
  }
}

export default Challenge;