import { Model, DataTypes } from 'sequelize';
import bcrypt from 'bcryptjs';

class User extends Model {
  static init(sequelize) {
    return super.init(
      {
        id: {
          type: DataTypes.UUID,
          defaultValue: DataTypes.UUIDV4,
          primaryKey: true
        },
        name: {
          type: DataTypes.STRING,
          allowNull: false,
          validate: {
            notEmpty: true,
            len: [2, 100]
          }
        },
        email: {
          type: DataTypes.STRING,
          allowNull: false,
          unique: true,
          validate: {
            isEmail: true
          }
        },
        password: {
          type: DataTypes.STRING,
          allowNull: false,
          validate: {
            notEmpty: true,
            len: [6, 100]
          }
        },
        role: {
          type: DataTypes.ENUM('admin', 'membershipUser', 'user'),
          defaultValue: 'user',
          allowNull: false
        },
        status: {
          type: DataTypes.ENUM('active', 'inactive', 'banned'),
          defaultValue: 'active',
          allowNull: false
        },
        avatarUrl: {
          type: DataTypes.STRING,
          allowNull: true
        },
        emailVerified: {
          type: DataTypes.BOOLEAN,
          defaultValue: false
        },
        verificationToken: {
          type: DataTypes.STRING,
          allowNull: true
        },
        resetPasswordToken: {
          type: DataTypes.STRING,
          allowNull: true
        },
        resetPasswordExpires: {
          type: DataTypes.DATE,
          allowNull: true
        },
        preferences: {
          type: DataTypes.JSON,
          defaultValue: {
            notifications: true,
            privacyLevel: 'public'
          }
        },
        lastLogin: {
          type: DataTypes.DATE,
          allowNull: true
        }
      },
      {
        sequelize,
        modelName: 'user',
        timestamps: true,
        hooks: {
          beforeCreate: async (user) => {
            if (user.password) {
              user.password = await bcrypt.hash(user.password, 10);
            }
          },
          beforeUpdate: async (user) => {
            if (user.changed('password')) {
              user.password = await bcrypt.hash(user.password, 10);
            }
          }
        }
      }
    );
  }

  // Define associations
  static associate(models) {
    this.hasMany(models.CTF, { foreignKey: 'creatorId', as: 'createdCTFs' });
    this.hasMany(models.Submission, { foreignKey: 'userId', as: 'submissions' });
    this.hasMany(models.Notification, { foreignKey: 'userId', as: 'notifications' });
    this.hasMany(models.Session, { foreignKey: 'userId', as: 'sessions' });
    this.hasMany(models.Achievement, { foreignKey: 'userId', as: 'achievements' });
    this.hasMany(models.Certificate, { foreignKey: 'userId', as: 'certificates' });
  }

  // Instance methods
  async comparePassword(candidatePassword) {
    return await bcrypt.compare(candidatePassword, this.password);
  }

  // Remove sensitive data when converting to JSON
  toJSON() {
    const values = { ...this.get() };
    delete values.password;
    delete values.verificationToken;
    delete values.resetPasswordToken;
    delete values.resetPasswordExpires;
    return values;
  }
}

export default User;