import express from 'express';
import { body } from 'express-validator';
import { 
  getDashboard,
  getUsers,
  updateUserRole,
  updateUserStatus,
  getPendingCTFs,
  updateCTFStatus,
  getSettings,
  updateSettings
} from '../controllers/admin.controller.js';
import { authenticate, authorize } from '../middleware/auth.js';
import { validate } from '../middleware/validation.js';

const router = express.Router();

// Apply admin authorization to all routes
router.use(authenticate, authorize('can_manage_users', 'can_manage_settings'));

// Admin dashboard
router.get('/dashboard', getDashboard);

// User management
router.get('/users', getUsers);

router.put(
  '/users/:userId/role',
  [
    body('role').notEmpty().withMessage('Role is required').isIn(['admin', 'membershipUser', 'user']).withMessage('Invalid role')
  ],
  validate,
  updateUserRole
);

router.put(
  '/users/:userId/status',
  [
    body('status').notEmpty().withMessage('Status is required').isIn(['active', 'banned']).withMessage('Status must be active or banned')
  ],
  validate,
  updateUserStatus
);

// CTF management
router.get('/ctfs/pending', getPendingCTFs);

router.put(
  '/ctfs/:ctfId/status',
  [
    body('status').notEmpty().withMessage('Status is required').isIn(['approved', 'rejected']).withMessage('Status must be approved or rejected'),
    body('feedback').optional().isString().withMessage('Feedback must be a string')
  ],
  validate,
  updateCTFStatus
);

// Settings management
router.get('/settings', getSettings);

router.put(
  '/settings',
  [
    body('registration.enabled').optional().isBoolean().withMessage('Enabled must be a boolean'),
    body('registration.requireApproval').optional().isBoolean().withMessage('RequireApproval must be a boolean'),
    body('registration.allowedDomains').optional().isArray().withMessage('AllowedDomains must be an array'),
    body('ctf.submissionLimit').optional().isInt({ min: 1 }).withMessage('SubmissionLimit must be a positive integer'),
    body('ctf.autoApproveTrusted').optional().isBoolean().withMessage('AutoApproveTrusted must be a boolean'),
    body('ctf.pointCalculation').optional().isIn(['dynamic', 'fixed']).withMessage('PointCalculation must be dynamic or fixed'),
    body('security.maxLoginAttempts').optional().isInt({ min: 1 }).withMessage('MaxLoginAttempts must be a positive integer'),
    body('security.sessionTimeout').optional().isInt({ min: 1 }).withMessage('SessionTimeout must be a positive integer'),
    body('security.ipWhitelist').optional().isArray().withMessage('IpWhitelist must be an array')
  ],
  validate,
  updateSettings
);

export default router;