import express from 'express';
import { body } from 'express-validator';
import { register, login, refreshToken, logout, verifyEmail, forgotPassword, resetPassword, registerAdmin, registerMembership } from '../controllers/auth.controller.js';
import { validate } from '../middleware/validation.js';
import { authenticate } from '../middleware/auth.js';

const router = express.Router();

// Register admin user
router.post(
  '/register-admin',
  [
    body('name').notEmpty().withMessage('Name is required').trim().isLength({ min: 2, max: 100 }).withMessage('Name must be between 2 and 100 characters'),
    body('email').notEmpty().withMessage('Email is required').isEmail().withMessage('Invalid email format').normalizeEmail(),
    body('password').notEmpty().withMessage('Password is required').isLength({ min: 6 }).withMessage('Password must be at least 6 characters long'),
    body('adminSecret').notEmpty().withMessage('Admin secret is required')
  ],
  validate,
  registerAdmin
);

// Register membership user
router.post(
  '/register-membership',
  [
    body('name').notEmpty().withMessage('Name is required').trim().isLength({ min: 2, max: 100 }).withMessage('Name must be between 2 and 100 characters'),
    body('email').notEmpty().withMessage('Email is required').isEmail().withMessage('Invalid email format').normalizeEmail(),
    body('password').notEmpty().withMessage('Password is required').isLength({ min: 6 }).withMessage('Password must be at least 6 characters long'),
    body('memberSecret').notEmpty().withMessage('Member secret is required')
  ],
  validate,
  registerMembership
);

// Register a new user
router.post(
  '/register',
  [
    body('name').notEmpty().withMessage('Name is required').trim().isLength({ min: 2, max: 100 }).withMessage('Name must be between 2 and 100 characters'),
    body('email').notEmpty().withMessage('Email is required').isEmail().withMessage('Invalid email format').normalizeEmail(),
    body('password').notEmpty().withMessage('Password is required').isLength({ min: 6 }).withMessage('Password must be at least 6 characters long')
  ],
  validate,
  register
);

// Login
router.post(
  '/login',
  [
    body('email').notEmpty().withMessage('Email is required').isEmail().withMessage('Invalid email format').normalizeEmail(),
    body('password').notEmpty().withMessage('Password is required')
  ],
  validate,
  login
);

// Refresh token
router.post(
  '/refresh-token',
  [
    body('refreshToken').notEmpty().withMessage('Refresh token is required')
  ],
  validate,
  refreshToken
);

// Logout
router.post('/logout', authenticate, logout);

// Verify email
router.get('/verify-email/:token', verifyEmail);

// Forgot password
router.post(
  '/forgot-password',
  [
    body('email').notEmpty().withMessage('Email is required').isEmail().withMessage('Invalid email format').normalizeEmail()
  ],
  validate,
  forgotPassword
);

// Reset password
router.post(
  '/reset-password/:token',
  [
    body('password').notEmpty().withMessage('Password is required').isLength({ min: 6 }).withMessage('Password must be at least 6 characters long')
  ],
  validate,
  resetPassword
);

export default router;