import express from 'express';
import { body, query } from 'express-validator';
import { 
  createChallenge, 
  getChallengeById, 
  updateChallenge, 
  deleteChallenge,
  submitFlag,
  getAllChallenges
} from '../controllers/challenge.controller.js';
import { authenticate, authorize, ownershipCheck } from '../middleware/auth.js';
import { validate } from '../middleware/validation.js';

const router = express.Router();

// Apply authentication to all routes
router.use(authenticate);

// Get all challenges with filtering and pagination
router.get(
  '/',
  [
    query('page').optional().isInt({ min: 1 }).toInt(),
    query('limit').optional().isInt({ min: 1, max: 100 }).toInt(),
    query('category').optional().trim(),
    query('difficulty').optional().trim(),
    query('search').optional().trim(),
    query('ctfId').optional().isUUID(4),
    query('sortBy').optional().isIn(['createdAt', 'title', 'points', 'solveCount', 'difficulty']),
    query('sortOrder').optional().isIn(['ASC', 'DESC'])
  ],
  validate,
  getAllChallenges
);

// Get challenge by ID
router.get('/:id', getChallengeById);

// Create a new challenge
router.post(
  '/',
  authorize('can_manage_challenges'),
  [
    body('ctfId').notEmpty().withMessage('CTF ID is required').isUUID().withMessage('Invalid CTF ID format'),
    body('title').notEmpty().withMessage('Title is required').trim().isLength({ min: 3, max: 200 }).withMessage('Title must be between 3 and 200 characters'),
    body('description').notEmpty().withMessage('Description is required'),
    body('category').notEmpty().withMessage('Category is required'),
    body('points').isInt({ min: 1 }).withMessage('Points must be a positive integer'),
    body('difficulty').isIn(['easy', 'medium', 'hard', 'extreme']).withMessage('Invalid difficulty level'),
    body('flag').notEmpty().withMessage('Flag is required'),
    body('videoLink').optional().isURL().withMessage('videoLink must be a valid URL'),
    body('image').optional().isURL().withMessage('image must be a valid URL')
  ],
  validate,
  createChallenge
);

// Update a challenge
router.put(
  '/:id',
  authorize('can_manage_challenges'),
  [
    body('title').optional().trim().isLength({ min: 3, max: 200 }).withMessage('Title must be between 3 and 200 characters'),
    body('description').optional(),
    body('category').optional(),
    body('points').optional().isInt({ min: 1 }).withMessage('Points must be a positive integer'),
    body('difficulty').optional().isIn(['easy', 'medium', 'hard', 'extreme']).withMessage('Invalid difficulty level'),
    body('flag').optional(),
    body('videoLink').optional().isURL().withMessage('videoLink must be a valid URL'),
    body('image').optional().isURL().withMessage('image must be a valid URL')
  ],
  validate,
  updateChallenge
);

// Delete a challenge
router.delete(
  '/:id',
  authorize('can_manage_challenges'),
  deleteChallenge
);

// Submit a flag for a challenge
router.post(
  '/:id/submit',
  [
    body('flag').notEmpty().withMessage('Flag is required')
  ],
  validate,
  submitFlag
);

export default router;