import express from 'express';
import { body } from 'express-validator';
import { 
  createCustomCTFPage,
  getCustomCTFPage,
  updateCustomCTFPage,
  deleteCustomCTFPage,
  getUserCustomCTFPages
} from '../controllers/customCTF.controller.js';
import { authenticate, authorize } from '../middleware/auth.js';
import { validate } from '../middleware/validation.js';

const router = express.Router();

// Apply authentication to all routes
router.use(authenticate);

// Create custom CTF page (requires can_create_ctf permission)
router.post(
  '/',
  authorize('can_create_ctf'),
  [
    body('title').notEmpty().withMessage('Title is required').trim().isLength({ min: 3, max: 200 }).withMessage('Title must be between 3 and 200 characters'),
    body('html').notEmpty().withMessage('HTML content is required'),
    body('css').notEmpty().withMessage('CSS content is required'),
    body('assets').optional().isArray().withMessage('Assets must be an array'),
    body('ctfId').optional().isUUID().withMessage('Invalid CTF ID format')
  ],
  validate,
  createCustomCTFPage
);

// Get custom CTF page by ID
router.get('/:id', getCustomCTFPage);

// Update custom CTF page
router.put(
  '/:id',
  authorize('can_create_ctf'),
  [
    body('title').optional().trim().isLength({ min: 3, max: 200 }).withMessage('Title must be between 3 and 200 characters'),
    body('html').optional(),
    body('css').optional(),
    body('assets').optional().isArray().withMessage('Assets must be an array'),
    body('ctfId').optional().isUUID().withMessage('Invalid CTF ID format')
  ],
  validate,
  updateCustomCTFPage
);

// Delete custom CTF page
router.delete('/:id', authorize('can_create_ctf'), deleteCustomCTFPage);

// Get all custom CTF pages for the authenticated user
router.get('/', getUserCustomCTFPages);

export default router;