import express from 'express';
import { body } from 'express-validator';
import { getProfile, updateProfile, updatePassword, updatePreferences } from '../controllers/profile.controller.js';
import { validate } from '../middleware/validation.js';
import multer from 'multer';
import path from 'path';
import { fileURLToPath } from 'url';
import fs from 'fs';

const router = express.Router();

// Get current directory
const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

// Define upload directory
const uploadDir = path.join(__dirname, '../../uploads/avatars');

// Ensure upload directory exists
if (!fs.existsSync(uploadDir)) {
  fs.mkdirSync(uploadDir, { recursive: true });
}

// Configure multer for file uploads
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    // Delete old avatar if exists
    if (req.user.avatarUrl) {
      const oldAvatarPath = path.join(__dirname, '../..', req.user.avatarUrl);
      if (fs.existsSync(oldAvatarPath)) {
        fs.unlinkSync(oldAvatarPath);
      }
    }
    cb(null, uploadDir);
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    const ext = path.extname(file.originalname);
    cb(null, `${req.user.id}-${uniqueSuffix}${ext}`);
  }
});

const fileFilter = (req, file, cb) => {
  // Accept only images
  if (file.mimetype.startsWith('image/')) {
    cb(null, true);
  } else {
    cb(new Error('Only image files are allowed'), false);
  }
};

const upload = multer({
  storage,
  limits: {
    fileSize: 2 * 1024 * 1024 // 2MB limit
  },
  fileFilter
});

// Error handling middleware for multer
const handleMulterError = (err, req, res, next) => {
  if (err instanceof multer.MulterError) {
    if (err.code === 'LIMIT_FILE_SIZE') {
      return res.status(400).json({
        status: 'error',
        message: 'File size cannot exceed 2MB'
      });
    }
    return res.status(400).json({
      status: 'error',
      message: 'Error uploading file'
    });
  } else if (err) {
    return res.status(400).json({
      status: 'error',
      message: err.message
    });
  }
  next();
};

// Get profile
router.get('/', getProfile);

// Update profile
router.put(
  '/',
  upload.single('avatar'),
  handleMulterError,
  [
    body('name').optional().trim().isLength({ min: 2, max: 100 }).withMessage('Name must be between 2 and 100 characters')
  ],
  validate,
  updateProfile
);

// Update password
router.put(
  '/password',
  [
    body('currentPassword').notEmpty().withMessage('Current password is required'),
    body('newPassword').notEmpty().withMessage('New password is required').isLength({ min: 6 }).withMessage('New password must be at least 6 characters long')
  ],
  validate,
  updatePassword
);

// Update preferences
router.put(
  '/preferences',
  [
    body('notifications').optional().isBoolean().withMessage('Notifications must be a boolean value'),
    body('privacyLevel').optional().isIn(['public', 'private', 'friends']).withMessage('Privacy level must be public, private, or friends')
  ],
  validate,
  updatePreferences
);

export default router;