import models from '../models/index.js';
import logger from '../utils/logger.js';
import { Op } from 'sequelize';

// Helper to get user rank
const getUserRank = async (userId) => {
  try {
    const users = await models.User.findAll({
      attributes: ['id'],
      include: [
        {
          model: models.Submission,
          as: 'submissions',
          where: { correct: true },
          required: false,
          attributes: ['pointsAwarded'],
        },
      ],
    });

    const userPoints = users.map(user => ({
      id: user.id,
      points: user.submissions?.reduce((sum, s) => sum + (s.pointsAwarded || 0), 0) || 0,
    }));

    userPoints.sort((a, b) => b.points - a.points);
    const rank = userPoints.findIndex(u => u.id === userId) + 1;
    return rank || userPoints.length;
  } catch (err) {
    logger.error('Error calculating user rank:', err);
    return null;
  }
};

// User Dashboard
export const getUserDashboard = async (req, res, next) => {
  try {
    const userId = req.user.id;

    // Recent submissions
    const recentSubmissions = await models.Submission.findAll({
      where: { userId },
      include: [
        {
          model: models.Challenge,
          as: 'challenge',
          include: [{ model: models.CTF, as: 'ctf' }],
        },
      ],
      order: [['createdAt', 'DESC']],
      limit: 5,
    });

    // Stats
    const [totalSolved, totalPoints] = await Promise.all([
      models.Submission.count({ where: { userId, correct: true } }),
      models.Submission.sum('pointsAwarded', { where: { userId, correct: true } }),
    ]);

    // Active CTFs
    const activeCTFs = await models.CTF.findAll({
      where: {
        status: 'approved',
        startDate: { [Op.lte]: new Date() },
        endDate: { [Op.gte]: new Date() },
      },
      include: [{ model: models.Challenge, as: 'challenges' }],
      limit: 5,
    });

    // Achievements
    const recentAchievements = await models.Achievement.findAll({
      where: { userId },
      order: [['earnedAt', 'DESC']],
      limit: 3,
    });

    return res.status(200).json({
      success: true,
      data: {
        stats: {
          totalSolved,
          totalPoints: totalPoints || 0,
          rank: await getUserRank(userId),
        },
        recentActivity: recentSubmissions.map(sub => ({
          id: sub.id,
          challengeTitle: sub.challenge?.title || 'Unknown',
          ctfTitle: sub.challenge?.ctf?.title || 'Unknown',
          correct: sub.correct,
          points: sub.pointsAwarded || 0,
          timestamp: sub.createdAt,
        })),
        activeCTFs: activeCTFs.map(ctf => ({
          id: ctf.id,
          title: ctf.title,
          endDate: ctf.endDate,
          totalChallenges: ctf.challenges?.length || 0,
        })),
        achievements: recentAchievements,
      },
    });
  } catch (err) {
    logger.error('Error in getUserDashboard:', err);
    next(err);
  }
};

// Member Dashboard
export const getMemberDashboard = async (req, res, next) => {
  try {
    const userId = req.user.id;

    const ctfs = await models.CTF.findAll({
      where: { creatorId: userId },
      include: [
        {
          model: models.Challenge,
          as: 'challenges',
          include: [
            {
              model: models.Submission,
              as: 'submissions',
            },
          ],
        },
      ],
    });

    let totalParticipants = 0;
    let totalSubmissions = 0;
    let totalSolves = 0;

    ctfs.forEach(ctf => {
      const participantSet = new Set();
      ctf.challenges?.forEach(challenge => {
        challenge.submissions?.forEach(sub => {
          participantSet.add(sub.userId);
          totalSubmissions++;
          if (sub.correct) totalSolves++;
        });
      });
      totalParticipants += participantSet.size;
    });

    const recentSubmissions = await models.Submission.findAll({
      include: [
        {
          model: models.Challenge,
          as: 'challenge',
          where: {
            ctfId: {
              [Op.in]: ctfs.map(c => c.id),
            },
          },
        },
        {
          model: models.User,
          as: 'user',
          attributes: ['id', 'name'],
        },
      ],
      order: [['createdAt', 'DESC']],
      limit: 10,
    });

    return res.status(200).json({
      success: true,
      data: {
        stats: {
          totalCTFs: ctfs.length,
          totalParticipants,
          totalSubmissions,
          solveRate: totalSubmissions > 0 ? (totalSolves / totalSubmissions) * 100 : 0,
        },
        ctfs: ctfs.map(ctf => ({
          id: ctf.id,
          title: ctf.title,
          status: ctf.status,
          startDate: ctf.startDate,
          endDate: ctf.endDate,
          challengeCount: ctf.challenges?.length || 0,
        })),
        recentActivity: recentSubmissions.map(sub => ({
          id: sub.id,
          user: {
            id: sub.user?.id,
            name: sub.user?.name,
          },
          challenge: {
            id: sub.challenge?.id,
            title: sub.challenge?.title,
          },
          correct: sub.correct,
          timestamp: sub.createdAt,
        })),
      },
    });
  } catch (err) {
    logger.error('Error in getMemberDashboard:', err);
    next(err);
  }
};

// Public Dashboard - accessible without authentication
export const getPublicDashboard = async (req, res, next) => {
  try {
    // Public stats
    const totalUsers = await models.User.count();
    const totalCTFs = await models.CTF.count();
    const totalChallenges = await models.Challenge.count();
    
    const activeCTFs = await models.CTF.count({
      where: {
        status: 'approved',
        startDate: {
          [Op.lte]: new Date()
        },
        endDate: {
          [Op.gte]: new Date()
        }
      }
    });

    // Featured CTFs (public ones)
    const featuredCTFs = await models.CTF.findAll({
      where: { status: 'approved' },
      limit: 5,
      order: [['createdAt', 'DESC']],
      attributes: ['id', 'title', 'description', 'startDate', 'endDate', 'difficulty', 'category']
    });

    return res.status(200).json({
      status: 'success',
      data: {
        stats: {
          totalUsers,
          totalCTFs,
          activeCTFs,
          totalChallenges
        },
        featuredCTFs
      }
    });
  } catch (err) {
    logger.error('Error in getPublicDashboard:', err);
    next(err);
  }
};