import models from '../models/index.js';
import { Op } from 'sequelize';
import logger from '../utils/logger.js';

// Get all notifications for the authenticated user
export const getNotifications = async (req, res, next) => {
  try {
    const userId = req.user.id;
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 10;
    const offset = (page - 1) * limit;

    // Get all notifications for the user
    const { count, rows } = await models.Notification.findAndCountAll({
      where: {
        userId,
        expiresAt: {
          [Op.or]: [
            { [Op.gt]: new Date() },
            { [Op.eq]: null }
          ]
        }
      },
      order: [['createdAt', 'DESC']],
      limit,
      offset
    });

    // Count unread notifications
    const unreadCount = await models.Notification.count({
      where: {
        userId,
        read: false,
        expiresAt: {
          [Op.or]: [
            { [Op.gt]: new Date() },
            { [Op.eq]: null }
          ]
        }
      }
    });

    return res.status(200).json({
      status: 'success',
      data: {
        notifications: rows,
        unreadCount,
        pagination: {
          total: count,
          currentPage: page,
          totalPages: Math.ceil(count / limit),
          limit
        }
      }
    });
  } catch (error) {
    next(error);
  }
};

// Mark a notification as read
export const markAsRead = async (req, res, next) => {
  try {
    const notificationId = req.params.id;
    
    // Update the notification
    const notification = await models.Notification.findByPk(notificationId);
    
    if (!notification) {
      return res.status(404).json({
        status: 'error',
        message: 'Notification not found'
      });
    }
    
    if (notification.userId !== req.user.id) {
      return res.status(403).json({
        status: 'error',
        message: 'You do not have permission to access this notification'
      });
    }
    
    notification.read = true;
    await notification.save();
    
    return res.status(200).json({
      status: 'success',
      message: 'Notification marked as read',
      data: notification
    });
  } catch (error) {
    next(error);
  }
};

// Mark all notifications as read
export const markAllAsRead = async (req, res, next) => {
  try {
    const userId = req.user.id;
    
    // Update all notifications for the user
    await models.Notification.update(
      { read: true },
      { 
        where: { 
          userId,
          read: false
        } 
      }
    );
    
    return res.status(200).json({
      status: 'success',
      message: 'All notifications marked as read'
    });
  } catch (error) {
    next(error);
  }
};

// Delete a notification
export const deleteNotification = async (req, res, next) => {
  try {
    const notificationId = req.params.id;
    
    // Find the notification
    const notification = await models.Notification.findByPk(notificationId);
    
    if (!notification) {
      return res.status(404).json({
        status: 'error',
        message: 'Notification not found'
      });
    }
    
    if (notification.userId !== req.user.id) {
      return res.status(403).json({
        status: 'error',
        message: 'You do not have permission to delete this notification'
      });
    }
    
    // Delete the notification
    await notification.destroy();
    
    return res.status(200).json({
      status: 'success',
      message: 'Notification deleted successfully'
    });
  } catch (error) {
    next(error);
  }
};