import models from '../models/index.js';
import { Op } from 'sequelize';
import logger from '../utils/logger.js';

// Get user progress and ranking
export const getUserProgress = async (req, res, next) => {
  try {
    const userId = req.user.id;
    
    // Get user data
    const user = await models.User.findByPk(userId);
    if (!user) {
      return res.status(404).json({
        status: 'error',
        message: 'User not found'
      });
    }
    
    // Get solved challenges
    const solvedChallenges = await models.Submission.count({
      where: {
        userId,
        correct: true
      }
    });
    
    // Calculate total points
    const pointsFromSubmissions = await models.Submission.sum('pointsAwarded', {
      where: {
        userId,
        correct: true
      }
    });
    
    const pointsFromAchievements = await models.Achievement.sum('points', {
      where: {
        userId
      }
    });
    
    const totalPoints = (pointsFromSubmissions || 0) + (pointsFromAchievements || 0);
    
    // Calculate rank (would be more complex in real application)
    const allUsers = await models.User.findAll({
      attributes: ['id']
    });
    
    const userPoints = [];
    for (const u of allUsers) {
      const userSubmissionPoints = await models.Submission.sum('pointsAwarded', {
        where: {
          userId: u.id,
          correct: true
        }
      });
      
      const userAchievementPoints = await models.Achievement.sum('points', {
        where: {
          userId: u.id
        }
      });
      
      userPoints.push({
        userId: u.id,
        points: (userSubmissionPoints || 0) + (userAchievementPoints || 0)
      });
    }
    
    userPoints.sort((a, b) => b.points - a.points);
    const userRank = userPoints.findIndex(u => u.userId === userId) + 1;
    
    // Get achievements
    const achievements = await models.Achievement.findAll({
      where: {
        userId
      },
      order: [['earnedAt', 'DESC']]
    });
    
    // Get skill progress
    const categories = await models.Challenge.findAll({
      attributes: [
        'category',
        [models.sequelize.fn('COUNT', models.sequelize.col('category')), 'count']
      ],
      include: [
        {
          model: models.Submission,
          as: 'submissions',
          where: {
            userId,
            correct: true
          },
          required: true
        }
      ],
      group: ['category']
    });
    
    const skillProgress = categories.map(category => {
      // This would be more complex in a real application
      const level = Math.floor(category.get('count') / 5) + 1; // Level up every 5 challenges
      const progress = (category.get('count') % 5) * 20; // Progress as percentage to next level
      
      return {
        category: category.category,
        level,
        progress,
        completedChallenges: category.get('count')
      };
    });
    
    return res.status(200).json({
      status: 'success',
      data: {
        currentRank: userRank,
        totalPoints,
        solvedChallenges,
        achievements: achievements.map(a => ({
          id: a.id,
          title: a.title,
          description: a.description,
          earnedAt: a.earnedAt
        })),
        skillProgress
      }
    });
  } catch (error) {
    next(error);
  }
};

// Get active challenges
export const getActiveChallenges = async (req, res, next) => {
  try {
    const userId = req.user.id;
    
    // Get ongoing CTFs
    const ongoingCTFs = await models.CTF.findAll({
      where: {
        status: 'approved',
        startDate: {
          [Op.lte]: new Date()
        },
        endDate: {
          [Op.gte]: new Date()
        }
      },
      include: [
        {
          model: models.Challenge,
          as: 'challenges'
        }
      ]
    });
    
    // Format CTFs with user progress
    const ctfsWithProgress = await Promise.all(ongoingCTFs.map(async (ctf) => {
      const totalChallenges = ctf.challenges.length;
      
      // Get completed challenges for this CTF
      const completedChallenges = await models.Submission.count({
        where: {
          userId,
          correct: true
        },
        include: [
          {
            model: models.Challenge,
            as: 'challenge',
            where: {
              ctfId: ctf.id
            },
            required: true
          }
        ]
      });
      
      // Calculate points earned in this CTF
      const pointsEarned = await models.Submission.sum('pointsAwarded', {
        where: {
          userId,
          correct: true
        },
        include: [
          {
            model: models.Challenge,
            as: 'challenge',
            where: {
              ctfId: ctf.id
            },
            required: true
          }
        ]
      });
      
      return {
        id: ctf.id,
        title: ctf.title,
        endTime: ctf.endDate,
        progress: {
          completed: completedChallenges,
          total: totalChallenges,
          points: pointsEarned || 0
        }
      };
    }));
    
    // Get recommended challenges
    const recommendedChallenges = await getRecommendedChallenges(userId);
    
    return res.status(200).json({
      status: 'success',
      data: {
        ongoingCTFs: ctfsWithProgress,
        recommendedChallenges
      }
    });
  } catch (error) {
    next(error);
  }
};

// Get learning paths
export const getLearningPaths = async (req, res, next) => {
  try {
    const userId = req.user.id;
    
    // Get all learning paths
    const learningPaths = await models.LearningPath.findAll({
      where: {
        isActive: true
      }
    });
    
    // Calculate progress for each learning path
    const pathsWithProgress = await Promise.all(learningPaths.map(async (path) => {
      // Extract challenge IDs from learning path steps
      const challengeIds = [];
      path.steps.forEach(step => {
        if (step.challengeId) {
          challengeIds.push(step.challengeId);
        }
      });
      
      // Count completed challenges in this path
      const completedChallenges = await models.Submission.count({
        where: {
          userId,
          correct: true,
          challengeId: {
            [Op.in]: challengeIds
          }
        }
      });
      
      const totalChallenges = challengeIds.length;
      const progress = totalChallenges > 0 ? (completedChallenges / totalChallenges) * 100 : 0;
      
      // Find next uncompleted challenge
      let nextChallenge = null;
      for (const step of path.steps) {
        if (step.challengeId) {
          const submission = await models.Submission.findOne({
            where: {
              userId,
              challengeId: step.challengeId,
              correct: true
            }
          });
          
          if (!submission) {
            const challenge = await models.Challenge.findByPk(step.challengeId);
            if (challenge) {
              nextChallenge = {
                id: challenge.id,
                title: challenge.title,
                estimatedTime: step.estimatedTime || 30 // Default 30 minutes
              };
              break;
            }
          }
        }
      }
      
      return {
        id: path.id,
        title: path.title,
        progress: Math.round(progress),
        nextChallenge
      };
    }));
    
    // Get certificates
    const certificates = await models.Certificate.findAll({
      where: {
        userId
      },
      order: [['issuedAt', 'DESC']]
    });
    
    return res.status(200).json({
      status: 'success',
      data: {
        paths: pathsWithProgress,
        certificates: certificates.map(cert => ({
          id: cert.id,
          title: cert.title,
          issuedAt: cert.issuedAt,
          validUntil: cert.validUntil
        }))
      }
    });
  } catch (error) {
    next(error);
  }
};

// Helper function to get recommended challenges
const getRecommendedChallenges = async (userId) => {
  try {
    // Get solved challenges
    const solvedChallenges = await models.Submission.findAll({
      where: {
        userId,
        correct: true
      },
      attributes: ['challengeId']
    });
    
    const solvedChallengeIds = solvedChallenges.map(s => s.challengeId);
    
    // Get user's preferred categories based on solved challenges
    const preferredCategories = await models.Challenge.findAll({
      where: {
        id: {
          [Op.in]: solvedChallengeIds
        }
      },
      attributes: [
        'category',
        [models.sequelize.fn('COUNT', models.sequelize.col('category')), 'count']
      ],
      group: ['category'],
      order: [[models.sequelize.literal('count'), 'DESC']],
      limit: 3
    });
    
    const categories = preferredCategories.map(c => c.category);
    
    // Find challenges that match user's level and preferred categories
    const challenges = await models.Challenge.findAll({
      where: {
        id: {
          [Op.notIn]: solvedChallengeIds
        },
        category: {
          [Op.in]: categories
        },
        isActive: true
      },
      include: [
        {
          model: models.CTF,
          as: 'ctf',
          where: {
            status: 'approved',
            startDate: {
              [Op.lte]: new Date()
            },
            endDate: {
              [Op.gte]: new Date()
            }
          },
          required: true
        }
      ],
      limit: 5
    });
    
    // Calculate completion rate for each challenge
    return await Promise.all(challenges.map(async (challenge) => {
      const totalAttempts = await models.Submission.count({
        where: {
          challengeId: challenge.id
        }
      });
      
      const correctAttempts = await models.Submission.count({
        where: {
          challengeId: challenge.id,
          correct: true
        }
      });
      
      const completionRate = totalAttempts > 0 ? (correctAttempts / totalAttempts) * 100 : 0;
      
      return {
        id: challenge.id,
        title: challenge.title,
        category: challenge.category,
        difficulty: challenge.difficulty,
        points: challenge.points,
        completionRate: Math.round(completionRate)
      };
    }));
  } catch (error) {
    logger.error('Error getting recommended challenges:', error);
    return [];
  }
};