import logger from '../utils/logger.js';

// Central error handling middleware
const errorHandler = (err, req, res, next) => {
  // Log the error with more context
  logger.error(`Error in ${req.method} ${req.originalUrl}:`, {
    error: {
      name: err.name,
      message: err.message,
      stack: err.stack
    },
    request: {
      method: req.method,
      url: req.originalUrl,
      body: req.body,
      params: req.params,
      query: req.query
    },
    user: req.user ? {
      id: req.user.id,
      role: req.user.role
    } : 'unauthenticated',
    ip: req.ip
  });

  // Default error status and message
  let statusCode = 500;
  let message = 'Internal Server Error';
  let errorData = {};

  // Handle different types of errors
  switch (err.name) {
    case 'SequelizeValidationError':
    case 'SequelizeUniqueConstraintError':
      statusCode = 400;
      message = 'Validation Error';
      errorData = {
        errors: err.errors.map(e => ({
          field: e.path,
          message: e.message
        }))
      };
      break;

    case 'ValidationError':
      statusCode = 400;
      message = err.message;
      break;

    case 'JsonWebTokenError':
    case 'TokenExpiredError':
      statusCode = 401;
      message = 'Invalid or expired token';
      break;

    case 'UnauthorizedError':
      statusCode = 401;
      message = err.message || 'Unauthorized';
      break;

    case 'ForbiddenError':
      statusCode = 403;
      message = err.message || 'Forbidden';
      break;

    case 'NotFoundError':
      statusCode = 404;
      message = err.message || 'Resource not found';
      break;

    case 'SequelizeDatabaseError':
      statusCode = 500;
      message = 'Database error occurred';
      if (process.env.NODE_ENV === 'development') {
        errorData.details = err.message;
      }
      break;

    case 'SequelizeConnectionError':
      statusCode = 503;
      message = 'Database connection error';
      break;

    case 'SequelizeTimeoutError':
      statusCode = 504;
      message = 'Database operation timed out';
      break;

    default:
      if (err.statusCode) {
        statusCode = err.statusCode;
        message = err.message;
      }
      break;
  }

  // In development, include additional error details
  if (process.env.NODE_ENV === 'development') {
    errorData.stack = err.stack;
    errorData.name = err.name;
  }

  // Send the error response
  res.status(statusCode).json({
    status: 'error',
    message,
    ...errorData
  });
};

// Handle uncaught exceptions
process.on('uncaughtException', (error) => {
  logger.error('Uncaught Exception:', {
    error: {
      name: error.name,
      message: error.message,
      stack: error.stack
    }
  });
  // Give the logger time to write before exiting
  setTimeout(() => {
    process.exit(1);
  }, 1000);
});

// Handle unhandled promise rejections
process.on('unhandledRejection', (reason, promise) => {
  logger.error('Unhandled Rejection at:', {
    promise,
    reason: reason instanceof Error ? {
      name: reason.name,
      message: reason.message,
      stack: reason.stack
    } : reason
  });
});

export default errorHandler;