import express from 'express';
import { body } from 'express-validator';
import { 
  createCTF, 
  getAllCTFs, 
  getCTFById, 
  updateCTF, 
  deleteCTF,
  getCTFsByUserId
} from '../controllers/ctf.controller.js';
import { authenticate, authorize, ownershipCheck } from '../middleware/auth.js';
import { validate } from '../middleware/validation.js';

const router = express.Router();

// Get all CTFs
router.get('/', authenticate, getAllCTFs);

// Get CTF by ID
router.get('/:id', authenticate, getCTFById);

// Create a new CTF (only for members and admins)
router.post(
  '/',
  authenticate,
  authorize('can_create_ctf'),
  [
    body('title').notEmpty().withMessage('Title is required').trim().isLength({ min: 3, max: 200 }).withMessage('Title must be between 3 and 200 characters'),
    body('description').notEmpty().withMessage('Description is required'),
    body('startDate').notEmpty().withMessage('Start date is required').isISO8601().withMessage('Start date must be a valid date'),
    body('endDate').notEmpty().withMessage('End date is required').isISO8601().withMessage('End date must be a valid date'),
    body('difficulty').notEmpty().withMessage('Difficulty is required').isIn(['beginner', 'intermediate', 'advanced', 'expert']).withMessage('Invalid difficulty level'),
    body('challenges').isArray({ min: 1 }).withMessage('At least one challenge is required'),
    body('challenges.*.title').notEmpty().withMessage('Challenge title is required'),
    body('challenges.*.description').notEmpty().withMessage('Challenge description is required'),
    body('challenges.*.category').notEmpty().withMessage('Challenge category is required'),
    body('challenges.*.points').isInt({ min: 1 }).withMessage('Challenge points must be a positive integer'),
    body('challenges.*.flag').notEmpty().withMessage('Challenge flag is required')
  ],
  validate,
  createCTF
);

// Update a CTF (only owner or admin)
router.put(
  '/:id',
  authenticate,
  authorize('can_create_ctf'),
  ownershipCheck('CTF', 'id', 'creatorId'),
  [
    body('title').optional().trim().isLength({ min: 3, max: 200 }).withMessage('Title must be between 3 and 200 characters'),
    body('description').optional(),
    body('startDate').optional().isISO8601().withMessage('Start date must be a valid date'),
    body('endDate').optional().isISO8601().withMessage('End date must be a valid date'),
    body('difficulty').optional().isIn(['beginner', 'intermediate', 'advanced', 'expert']).withMessage('Invalid difficulty level')
  ],
  validate,
  updateCTF
);
router.get('/user/:userId', authenticate, getCTFsByUserId);
// Delete a CTF (only owner or admin)
router.delete(
  '/:id',
  authenticate,
  authorize('can_create_ctf'),
  ownershipCheck('CTF', 'id', 'creatorId'),
  deleteCTF
);

export default router;