import models from '../models/index.js';
import { sequelize } from '../models/index.js';
import { Op } from 'sequelize';
import logger from '../utils/logger.js';
import { sendEmail, emailTemplates } from '../utils/email.js';

// Get admin dashboard data
export const getDashboard = async (req, res, next) => {
  try {
    // System stats
    const totalUsers = await models.User.count();
    
    const activeUsers24h = await models.User.count({
      where: {
        lastLogin: {
          [Op.gte]: new Date(Date.now() - 24 * 60 * 60 * 1000)
        }
      }
    });
    
    const totalCTFs = await models.CTF.count();
    
    const activeCTFs = await models.CTF.count({
      where: {
        status: 'approved',
        startDate: {
          [Op.lte]: new Date()
        },
        endDate: {
          [Op.gte]: new Date()
        }
      }
    });
    
    const totalChallenges = await models.Challenge.count();
    
    const totalSubmissions24h = await models.Submission.count({
      where: {
        createdAt: {
          [Op.gte]: new Date(Date.now() - 24 * 60 * 60 * 1000)
        }
      }
    });

    // Recent activities
    const recentActivities = await getRecentActivities();

    // System health
    const serverStatus = 'healthy'; // In a real application, this would come from monitoring
    const responseTime = 100; // In a real application, this would be measured
    const activeSessions = await models.Session.count({
      where: {
        isRevoked: false,
        expiresAt: {
          [Op.gt]: new Date()
        }
      }
    });

    // Format the response
    const dashboardData = {
      stats: {
        totalUsers,
        activeUsers24h,
        totalCTFs,
        activeCTFs,
        totalChallenges,
        totalSubmissions24h
      },
      recentActivities,
      systemHealth: {
        serverStatus,
        responseTime,
        activeSessions,
        storageUsage: 0 // In a real application, this would be calculated
      }
    };

    return res.status(200).json({
      status: 'success',
      data: dashboardData
    });
  } catch (error) {
    next(error);
  }
};

// Get all users with pagination and filtering
export const getUsers = async (req, res, next) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 10;
    const offset = (page - 1) * limit;
    const role = req.query.role;
    const status = req.query.status;
    const search = req.query.search;

    // Build query conditions
    const whereConditions = {};
    
    if (role) {
      whereConditions.role = role;
    }
    
    if (status) {
      whereConditions.status = status;
    }
    
    if (search) {
      whereConditions[Op.or] = [
        {
          name: {
            [Op.like]: `%${search}%`
          }
        },
        {
          email: {
            [Op.like]: `%${search}%`
          }
        }
      ];
    }

    // Get users with pagination
    const { count, rows } = await models.User.findAndCountAll({
      where: whereConditions,
      attributes: { exclude: ['password', 'verificationToken', 'resetPasswordToken', 'resetPasswordExpires'] },
      order: [['createdAt', 'DESC']],
      limit,
      offset
    });

    return res.status(200).json({
      status: 'success',
      data: {
        users: rows,
        pagination: {
          total: count,
          currentPage: page,
          totalPages: Math.ceil(count / limit),
          limit
        }
      }
    });
  } catch (error) {
    next(error);
  }
};

// Update user role
export const updateUserRole = async (req, res, next) => {
  try {
    const { userId } = req.params;
    const { role, reason } = req.body;

    // Find user
    const user = await models.User.findByPk(userId);
    if (!user) {
      return res.status(404).json({
        status: 'error',
        message: 'User not found'
      });
    }

    // Update role
    const oldRole = user.role;
    user.role = role;
    await user.save();

    // Log the role change
    logger.info(`User role changed: ${userId} from ${oldRole} to ${role} by ${req.user.id}. Reason: ${reason || 'Not provided'}`);

    // Create a notification for the user
    await models.Notification.create({
      userId,
      type: 'role_change',
      message: `Your account role has been changed from ${oldRole} to ${role}.`,
      read: false
    });

    return res.status(200).json({
      status: 'success',
      message: 'User role updated successfully',
      data: {
        user: {
          id: user.id,
          role: user.role,
          updatedAt: user.updatedAt
        }
      }
    });
  } catch (error) {
    next(error);
  }
};

// Update user status
export const updateUserStatus = async (req, res, next) => {
  try {
    const { userId } = req.params;
    const { status, reason } = req.body;

    // Find user
    const user = await models.User.findByPk(userId);
    if (!user) {
      return res.status(404).json({
        status: 'error',
        message: 'User not found'
      });
    }

    // Cannot update own status
    if (userId === req.user.id) {
      return res.status(403).json({
        status: 'error',
        message: 'You cannot update your own status'
      });
    }

    // Update status
    const oldStatus = user.status;
    user.status = status;
    await user.save();

    // If user is banned, revoke all sessions
    if (status === 'banned') {
      await models.Session.update(
        { isRevoked: true },
        { where: { userId } }
      );
    }

    // Log the status change
    logger.info(`User status changed: ${userId} from ${oldStatus} to ${status} by ${req.user.id}. Reason: ${reason || 'Not provided'}`);

    // Create a notification for the user
    await models.Notification.create({
      userId,
      type: 'status_change',
      message: `Your account status has been changed to ${status}.`,
      read: false
    });

    return res.status(200).json({
      status: 'success',
      message: 'User status updated successfully',
      data: {
        user: {
          id: user.id,
          status: user.status,
          updatedAt: user.updatedAt
        }
      }
    });
  } catch (error) {
    next(error);
  }
};

// Get pending CTFs
export const getPendingCTFs = async (req, res, next) => {
  try {
    // Get all pending CTFs
    const pendingCTFs = await models.CTF.findAll({
      where: {
        status: 'pending'
      },
      include: [
        {
          model: models.User,
          as: 'creator',
          attributes: ['id', 'name', 'email']
        }
      ],
      order: [['createdAt', 'ASC']]
    });

    return res.status(200).json({
      status: 'success',
      data: {
        ctfs: pendingCTFs
      }
    });
  } catch (error) {
    next(error);
  }
};

// Update CTF status
export const updateCTFStatus = async (req, res, next) => {
  try {
    const { ctfId } = req.params;
    const { status, feedback } = req.body;

    // Find CTF
    const ctf = await models.CTF.findByPk(ctfId, {
      include: [
        {
          model: models.User,
          as: 'creator',
          attributes: ['id', 'name', 'email']
        }
      ]
    });

    if (!ctf) {
      return res.status(404).json({
        status: 'error',
        message: 'CTF not found'
      });
    }

    // Update status
    ctf.status = status;
    if (feedback) {
      ctf.reviewFeedback = feedback;
    }
    await ctf.save();

    // Create a notification for the creator
    await models.Notification.create({
      userId: ctf.creatorId,
      type: 'ctf_review',
      message: `Your CTF "${ctf.title}" has been ${status}.`,
      read: false,
      actionUrl: `/ctfs/${ctf.id}`
    });

    // If approved, send email to creator
    if (status === 'approved') {
      const ctfUrl = `${process.env.FRONTEND_URL}/ctfs/${ctf.id}`;
      await sendEmail({
        to: ctf.creator.email,
        ...emailTemplates.ctfApproved(ctf.creator.name, ctf.title, ctfUrl)
      });
    }

    return res.status(200).json({
      status: 'success',
      message: `CTF ${status} successfully`,
      data: {
        ctf: {
          id: ctf.id,
          title: ctf.title,
          status: ctf.status,
          updatedAt: ctf.updatedAt
        }
      }
    });
  } catch (error) {
    next(error);
  }
};

// Get application settings
export const getSettings = async (req, res, next) => {
  try {
    // Get all settings
    const settings = await models.Setting.findAll({
      order: [['category', 'ASC'], ['key', 'ASC']]
    });

    // Group settings by category
    const groupedSettings = settings.reduce((result, setting) => {
      if (!result[setting.category]) {
        result[setting.category] = {};
      }
      result[setting.category][setting.key] = setting.value;
      return result;
    }, {});

    return res.status(200).json({
      status: 'success',
      data: groupedSettings
    });
  } catch (error) {
    next(error);
  }
};

// Update application settings
export const updateSettings = async (req, res, next) => {
  const transaction = await sequelize.transaction();

  try {
    const updatedSettings = req.body;

    // Update each setting
    for (const category in updatedSettings) {
      for (const key in updatedSettings[category]) {
        const fullKey = `${category}.${key}`;
        const value = updatedSettings[category][key];

        // Find or create the setting
        const [setting, created] = await models.Setting.findOrCreate({
          where: { key: fullKey },
          defaults: {
            key: fullKey,
            value,
            category
          },
          transaction
        });

        // If setting exists, update it
        if (!created) {
          setting.value = value;
          await setting.save({ transaction });
        }
      }
    }

    // Commit transaction
    await transaction.commit();

    return res.status(200).json({
      status: 'success',
      message: 'Settings updated successfully'
    });
  } catch (error) {
    // Rollback transaction
    await transaction.rollback();
    next(error);
  }
};

// Helper function to get recent activities
const getRecentActivities = async () => {
  try {
    // Get recent registrations
    const registrations = await models.User.findAll({
      attributes: ['id', 'name', 'createdAt'],
      order: [['createdAt', 'DESC']],
      limit: 5
    });

    // Get recent submissions
    const submissions = await models.Submission.findAll({
      attributes: ['id', 'userId', 'createdAt', 'correct'],
      include: [
        {
          model: models.User,
          as: 'user',
          attributes: ['id', 'name']
        },
        {
          model: models.Challenge,
          as: 'challenge',
          attributes: ['id', 'title']
        }
      ],
      order: [['createdAt', 'DESC']],
      limit: 5
    });

    // Get recent CTF creations
    const ctfCreations = await models.CTF.findAll({
      attributes: ['id', 'title', 'creatorId', 'createdAt'],
      include: [
        {
          model: models.User,
          as: 'creator',
          attributes: ['id', 'name']
        }
      ],
      order: [['createdAt', 'DESC']],
      limit: 5
    });

    // Combine and format activities
    const activities = [
      ...registrations.map(reg => ({
        id: `reg_${reg.id}`,
        type: 'registration',
        user: { id: reg.id, name: reg.name },
        details: 'Registered a new account',
        timestamp: reg.createdAt
      })),
      ...submissions.map(sub => ({
        id: `sub_${sub.id}`,
        type: 'submission',
        user: { id: sub.user.id, name: sub.user.name },
        details: `${sub.correct ? 'Solved' : 'Attempted'} challenge "${sub.challenge.title}"`,
        timestamp: sub.createdAt
      })),
      ...ctfCreations.map(ctf => ({
        id: `ctf_${ctf.id}`,
        type: 'ctf_creation',
        user: { id: ctf.creator.id, name: ctf.creator.name },
        details: `Created CTF "${ctf.title}"`,
        timestamp: ctf.createdAt
      }))
    ];

    // Sort by timestamp (most recent first)
    activities.sort((a, b) => new Date(b.timestamp) - new Date(a.timestamp));

    // Return the 10 most recent activities
    return activities.slice(0, 10);
  } catch (error) {
    logger.error('Error getting recent activities:', error);
    return [];
  }
};