import models from '../models/index.js';
import { sendEmail, emailTemplates } from '../utils/email.js';
import logger from '../utils/logger.js';
import { sequelize } from '../models/index.js';
import { Op } from 'sequelize';

// Create a new CTF
export const createCTF = async (req, res, next) => {
  let transaction;
  
  try {
    transaction = await sequelize.transaction();
    
    const { title, description, startDate, endDate, difficulty, challenges } = req.body;
    const creatorId = req.user.id;
    
    // Validate required fields
    if (!title || !description || !startDate || !endDate || !difficulty || !challenges) {
      return res.status(400).json({
        status: 'error',
        message: 'Missing required fields'
      });
    }

    // Validate challenges array
    if (!Array.isArray(challenges) || challenges.length === 0) {
      return res.status(400).json({
        status: 'error',
        message: 'At least one challenge is required'
      });
    }

    // Check if user is a member or admin
    if (!['admin', 'membershipUser'].includes(req.user.role)) {
      return res.status(403).json({
        status: 'error',
        message: 'Only members and admins can create CTFs'
      });
    }
    
    // Create CTF
    const ctf = await models.CTF.create({
      title,
      description,
      startDate: new Date(startDate),
      endDate: new Date(endDate),
      difficulty,
      creatorId,
      status: req.user.role === 'admin' ? 'approved' : 'pending',
      visibility: 'public',
      tags: []
    }, { transaction });
    
    // Create challenges
    const createdChallenges = await Promise.all(challenges.map(async (challenge, index) => {
      // Validate challenge fields
      if (!challenge.title || !challenge.description || !challenge.category || !challenge.points || !challenge.flag) {
        throw new Error(`Invalid challenge data at index ${index}`);
      }

      return await models.Challenge.create({
        ctfId: ctf.id,
        title: challenge.title,
        description: challenge.description,
        category: challenge.category,
        points: challenge.points,
        difficulty: challenge.difficulty || 'medium',
        flag: challenge.flag,
        hints: challenge.hints || [],
        files: challenge.files || [],
        videoLink: videoLink || null,
        image: image || null, 
        order: index + 1,
        isActive: true
      }, { transaction });
    }));
    
    // Commit transaction
    await transaction.commit();
    
    // Create notifications
    if (req.user.role === 'admin') {
      // Create notification for auto-approved CTF
      await models.Notification.create({
        userId: creatorId,
        type: 'ctf_approved',
        message: `Your CTF "${title}" has been automatically approved.`,
        read: false,
        actionUrl: `/ctfs/${ctf.id}`
      });
    } else {
      // Notify admins about pending CTF
      const admins = await models.User.findAll({
        where: {
          role: 'admin'
        }
      });
      
      for (const admin of admins) {
        await models.Notification.create({
          userId: admin.id,
          type: 'ctf_pending',
          message: `New CTF "${title}" is pending approval.`,
          read: false,
          actionUrl: `/admin/ctfs/pending`
        });
      }
    }
    
    return res.status(201).json({
      status: 'success',
      message: req.user.role === 'admin' ? 'CTF created successfully' : 'CTF submitted for approval',
      data: {
        id: ctf.id,
        status: ctf.status,
        challenges: createdChallenges.map(c => ({
          id: c.id,
          title: c.title
        }))
      }
    });
  } catch (error) {
    // Rollback transaction if it exists
    if (transaction) await transaction.rollback();
    
    // Log the error
    logger.error('Error creating CTF:', {
      error: error.message,
      stack: error.stack,
      userId: req.user?.id,
      body: req.body
    });

    // Send appropriate error response
    if (error.name === 'SequelizeValidationError') {
      return res.status(400).json({
        status: 'error',
        message: 'Validation error',
        errors: error.errors.map(e => ({
          field: e.path,
          message: e.message
        }))
      });
    }

    if (error.message.includes('Invalid challenge data')) {
      return res.status(400).json({
        status: 'error',
        message: error.message
      });
    }

    next(error);
  }
};

// Get all CTFs
export const getAllCTFs = async (req, res, next) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 10;
    const offset = (page - 1) * limit;
    const status = req.query.status;
    const difficulty = req.query.difficulty;
    const search = req.query.search;
    
    // Build query conditions
    const whereConditions = {
      status: 'approved'
    };
    
    // If admin or member, show own CTFs regardless of status
    if (['admin', 'membershipUser'].includes(req.user.role)) {
      if (status) {
        whereConditions.status = status;
      } else {
        delete whereConditions.status;
      }
    }
    
    if (difficulty) {
      whereConditions.difficulty = difficulty;
    }
    
    if (search) {
      whereConditions[Op.or] = [
        {
          name: {
            [Op.like]: `%${search}%`
          }
        },
        {
          description: {
            [Op.like]: `%${search}%`
          }
        }
      ];
    }
    
    // Get CTFs with pagination
    const { count, rows } = await models.CTF.findAndCountAll({
      where: whereConditions,
      include: [
        {
          model: models.User,
          as: 'creator',
          attributes: ['id', 'name']
        },
        {
          model: models.Challenge,
          as: 'challenges',
          attributes: ['id', 'title', 'description', 'category', 'points', 'difficulty', 'hints', 'files', 'order', 'videoLink', 'image'],
        }
      ],
      order: [['createdAt', 'DESC']],
      limit,
      offset
    });
    
    // Format CTFs
    const formattedCTFs = rows.map(ctf => {
      const isActive = new Date() >= new Date(ctf.startDate) && new Date() <= new Date(ctf.endDate);
      const totalPoints = ctf.challenges.reduce((sum, challenge) => sum + challenge.points, 0);
      const totalChallenges = ctf.challenges.length;
      
      return {
        id: ctf.id,
        title: ctf.title,
        description: ctf.description,
        startDate: ctf.startDate,
        endDate: ctf.endDate,
        difficulty: ctf.difficulty,
        creator: ctf.creator ? {
          id: ctf.creator.id,
          name: ctf.creator.name
        } : null,
        status: ctf.status,
        isActive,
        stats: {
          totalPoints,
          totalChallenges,
          categories: [...new Set(ctf.challenges.map(c => c.category))]
        }
      };
    });
    
    return res.status(200).json({
      status: 'success',
      data: {
        ctfs: formattedCTFs,
        pagination: {
          total: count,
          currentPage: page,
          totalPages: Math.ceil(count / limit),
          limit
        }
      }
    });
  } catch (error) {
    next(error);
  }
};
// Get CTFs by User ID (only accessible by admin or the user themself)
export const getCTFsByUserId = async (req, res, next) => {
  try {
    const { userId } = req.params;

    // Only admins or the user themself can fetch this
    if (req.user.role !== 'admin' && req.user.id !== userId) {
      return res.status(403).json({
        status: 'error',
        message: 'You are not authorized to view these CTFs'
      });
    }

    const ctfs = await models.CTF.findAll({
      where: {
        creatorId: userId
      },
      include: [
        {
          model: models.Challenge,
          as: 'challenges',
          attributes: ['id', 'title', 'description', 'category', 'points', 'difficulty', 'hints', 'files', 'order', 'videoLink', 'image'],
        }
      ],
      order: [['createdAt', 'DESC']]
    });

    const formattedCTFs = ctfs.map(ctf => {
      const isActive = new Date() >= new Date(ctf.startDate) && new Date() <= new Date(ctf.endDate);
      const totalPoints = ctf.challenges.reduce((sum, challenge) => sum + challenge.points, 0);
      const totalChallenges = ctf.challenges.length;

      return {
        id: ctf.id,
        title: ctf.title,
        description: ctf.description,
        startDate: ctf.startDate,
        endDate: ctf.endDate,
        difficulty: ctf.difficulty,
        status: ctf.status,
        isActive,
        stats: {
          totalPoints,
          totalChallenges,
          categories: [...new Set(ctf.challenges.map(c => c.category))]
        }
      };
    });

    return res.status(200).json({
      status: 'success',
      data: {
        ctfs: formattedCTFs
      }
    });
  } catch (error) {
    next(error);
  }
};
// Get CTF by ID
export const getCTFById = async (req, res, next) => {
  try {
    const { id } = req.params;
    
    // Get CTF
    const ctf = await models.CTF.findByPk(id, {
      include: [
        {
          model: models.User,
          as: 'creator',
          attributes: ['id', 'name']
        },
        {
          model: models.Challenge,
          as: 'challenges',
          attributes: ['id', 'title', 'description', 'category', 'points', 'difficulty', 'hints', 'files', 'order', 'videoLink', 'image'],
        }
      ]
    });
    
    if (!ctf) {
      return res.status(404).json({
        status: 'error',
        message: 'CTF not found'
      });
    }
    
    // Check if user can view this CTF
    const canView = 
      ctf.status === 'approved' || 
      req.user.role === 'admin' || 
      (req.user.role === 'membershipUser' && ctf.creatorId === req.user.id);
    
    if (!canView) {
      return res.status(403).json({
        status: 'error',
        message: 'You do not have permission to view this CTF'
      });
    }
    
    // For challenges, hide flags unless user is the creator or admin
    const isCreatorOrAdmin = ctf.creatorId === req.user.id || req.user.role === 'admin';
    
    // Get user's solved challenges
    const solvedChallenges = await models.Submission.findAll({
      where: {
        userId: req.user.id,
        correct: true,
        challengeId: {
          [Op.in]: ctf.challenges.map(c => c.id)
        }
      },
      attributes: ['challengeId']
    });
    
    const solvedChallengeIds = solvedChallenges.map(s => s.challengeId);
    
    // Get participation stats
    const participants = await models.Submission.findAll({
      where: {
        challengeId: {
          [Op.in]: ctf.challenges.map(c => c.id)
        },
        correct: true
      },
      attributes: [
        'userId',
        [sequelize.fn('SUM', sequelize.col('pointsAwarded')), 'total_points']
      ],
      group: ['userId'],
      order: [[sequelize.col('total_points'), 'DESC']],
      limit: 10
    });
    
    const participantIds = participants.map(p => p.userId);
    const participantUsers = await models.User.findAll({
      where: {
        id: {
          [Op.in]: participantIds
        }
      },
      attributes: ['id', 'name']
    });
    
    const participantMap = participantUsers.reduce((map, user) => {
      map[user.id] = user;
      return map;
    }, {});
    
    const leaderboard = participants.map(p => ({
      user: {
        id: p.userId,
        name: participantMap[p.userId]?.name || 'Unknown'
      },
      points: parseInt(p.get('total_points')) || 0
    }));
    
    // Format the response
    const formattedCTF = {
      id: ctf.id,
      title: ctf.title,
      description: ctf.description,
      startDate: ctf.startDate,
      endDate: ctf.endDate,
      difficulty: ctf.difficulty,
      creator: ctf.creator ? {
        id: ctf.creator.id,
        name: ctf.creator.name
      } : null,
      status: ctf.status,
      isActive: new Date() >= new Date(ctf.startDate) && new Date() <= new Date(ctf.endDate),
      challenges: ctf.challenges.map(challenge => {
        const isSolved = solvedChallengeIds.includes(challenge.id);
        
        return {
          id: challenge.id,
          title: challenge.title,
          description: challenge.description,
          category: challenge.category,
          points: challenge.points,
          difficulty: challenge.difficulty,
          hints: challenge.hints,
          files: challenge.files,
          order: challenge.order,
          videoLink: challenge.videoLink,
          image: challenge.image,
          isSolved,
          // Only include flag for creator or admin
          ...(isCreatorOrAdmin && { flag: challenge.flag })
        };
      }),
      leaderboard
    };
    
    return res.status(200).json({
      status: 'success',
      data: formattedCTF
    });
  } catch (error) {
    next(error);
  }
};

// Update a CTF
export const updateCTF = async (req, res, next) => {
  try {
    const { id } = req.params;
    const { title, description, startDate, endDate, difficulty } = req.body;
    
    // Get CTF
    const ctf = await models.CTF.findByPk(id);
    
    if (!ctf) {
      return res.status(404).json({
        status: 'error',
        message: 'CTF not found'
      });
    }
    
    // Check if user can update this CTF
    const canUpdate = 
      req.user.role === 'admin' || 
      (req.user.role === 'membershipUser' && ctf.creatorId === req.user.id);
    
    if (!canUpdate) {
      return res.status(403).json({
        status: 'error',
        message: 'You do not have permission to update this CTF'
      });
    }
    
    // Update CTF
    if (title) ctf.title = title;
    if (description) ctf.description = description;
    if (startDate) ctf.startDate = startDate;
    if (endDate) ctf.endDate = endDate;
    if (difficulty) ctf.difficulty = difficulty;
    
    // If not admin and CTF was already approved, set status back to pending
    if (req.user.role !== 'admin' && ctf.status === 'approved') {
      ctf.status = 'pending';
    }
    
    await ctf.save();
    
    // If status changed to pending, notify admins
    if (ctf.status === 'pending') {
      const admins = await models.User.findAll({
        where: {
          role: 'admin'
        }
      });
      
      for (const admin of admins) {
        await models.Notification.create({
          userId: admin.id,
          type: 'ctf_update',
          message: `CTF "${ctf.title}" has been updated and needs approval.`,
          read: false,
          actionUrl: `/admin/ctfs/pending`
        });
      }
    }
    
    return res.status(200).json({
      status: 'success',
      message: 'CTF updated successfully',
      data: {
        id: ctf.id,
        title: ctf.title,
        status: ctf.status
      }
    });
  } catch (error) {
    next(error);
  }
};

// Delete a CTF
export const deleteCTF = async (req, res, next) => {
  let transaction;
  
  try {
    transaction = await sequelize.transaction();
    
    const { id } = req.params;
    
    // Get CTF
    const ctf = await models.CTF.findByPk(id, {
      include: [
        {
          model: models.Challenge,
          as: 'challenges'
        }
      ],
      transaction
    });
    
    if (!ctf) {
      await transaction.rollback();
      return res.status(404).json({
        status: 'error',
        message: 'CTF not found'
      });
    }
    
    // Check if user can delete this CTF
    const canDelete = 
      req.user.role === 'admin' || 
      (req.user.role === 'membershipUser' && ctf.creatorId === req.user.id);
    
    if (!canDelete) {
      await transaction.rollback();
      return res.status(403).json({
        status: 'error',
        message: 'You do not have permission to delete this CTF'
      });
    }
    
    // Delete submissions for challenges in this CTF
    if (ctf.challenges && ctf.challenges.length > 0) {
      await models.Submission.destroy({
        where: {
          challengeId: {
            [Op.in]: ctf.challenges.map(c => c.id)
          }
        },
        transaction
      });
    }
    
    // Delete challenges
    await models.Challenge.destroy({
      where: {
        ctfId: id
      },
      transaction
    });
    
    // Delete CTF
    await ctf.destroy({ transaction });
    
    // Commit transaction
    await transaction.commit();
    
    return res.status(200).json({
      status: 'success',
      message: 'CTF deleted successfully'
    });
  } catch (error) {
    // Rollback transaction if it exists
    if (transaction) await transaction.rollback();
    
    // Log the error
    logger.error('Error deleting CTF:', {
      error: error.message,
      stack: error.stack,
      userId: req.user?.id,
      ctfId: req.params.id
    });

    next(error);
  }
}