import models from '../models/index.js';
import { Op } from 'sequelize';
import logger from '../utils/logger.js';

// Get user profile
export const getProfile = async (req, res, next) => {
  try {
    // Get user with associated data
    const user = await models.User.findByPk(req.user.id, {
      attributes: { exclude: ['password', 'verificationToken', 'resetPasswordToken', 'resetPasswordExpires'] },
      include: [
        {
          model: models.Achievement,
          as: 'achievements',
          limit: 5,
          order: [['earnedAt', 'DESC']]
        }
      ]
    });

    if (!user) {
      return res.status(404).json({
        status: 'error',
        message: 'User not found'
      });
    }

    // Get statistics for the user
    const submissionCount = await models.Submission.count({
      where: {
        userId: user.id,
        correct: true
      }
    });

    // Calculate user rank
    const userRank = await getUserRank(user.id);

    // Calculate total points
    const totalPoints = await calculateUserPoints(user.id);

    // Format the response
    const profile = {
      id: user.id,
      name: user.name,
      email: user.email,
      avatarUrl: user.avatarUrl,
      role: user.role,
      joinedDate: user.createdAt,
      stats: {
        rank: userRank,
        points: totalPoints,
        solvedChallenges: submissionCount
      },
      achievements: user.achievements.map(achievement => ({
        id: achievement.id,
        title: achievement.title,
        description: achievement.description,
        iconUrl: achievement.iconUrl,
        earnedAt: achievement.earnedAt
      })),
      preferences: user.preferences
    };

    return res.status(200).json({
      status: 'success',
      data: profile
    });
  } catch (error) {
    next(error);
  }
};

// Update user profile
export const updateProfile = async (req, res, next) => {
  try {
    const { name } = req.body;
    const userId = req.user.id;

    // Find user
    const user = await models.User.findByPk(userId);
    if (!user) {
      return res.status(404).json({
        status: 'error',
        message: 'User not found'
      });
    }

    // Update user fields
    if (name) {
      user.name = name;
    }

    // Handle avatar upload
    if (req.file) {
      // For local storage, use relative path
      const avatarUrl = `/uploads/avatars/${req.file.filename}`;
      user.avatarUrl = avatarUrl;
    }

    // Save user
    await user.save();

    return res.status(200).json({
      status: 'success',
      message: 'Profile updated successfully',
      data: {
        id: user.id,
        name: user.name,
        email: user.email,
        avatarUrl: user.avatarUrl ? `${process.env.BASE_URL}${user.avatarUrl}` : null
      }
    });
  } catch (error) {
    next(error);
  }
};

// Update user password
export const updatePassword = async (req, res, next) => {
  try {
    const { currentPassword, newPassword } = req.body;
    const userId = req.user.id;

    // Find user
    const user = await models.User.findByPk(userId);
    if (!user) {
      return res.status(404).json({
        status: 'error',
        message: 'User not found'
      });
    }

    // Verify current password
    const isPasswordValid = await user.comparePassword(currentPassword);
    if (!isPasswordValid) {
      return res.status(401).json({
        status: 'error',
        message: 'Current password is incorrect'
      });
    }

    // Update password
    user.password = newPassword;
    await user.save();

    // Revoke all sessions except the current one
    const authHeader = req.headers.authorization;
    const token = authHeader.split(' ')[1];
    
    await models.Session.update(
      { isRevoked: true },
      {
        where: {
          userId: user.id,
          refreshToken: {
            [Op.ne]: token
          }
        }
      }
    );

    return res.status(200).json({
      status: 'success',
      message: 'Password updated successfully'
    });
  } catch (error) {
    next(error);
  }
};

// Update user preferences
export const updatePreferences = async (req, res, next) => {
  try {
    const { notifications, privacyLevel } = req.body;
    const userId = req.user.id;

    // Find user
    const user = await models.User.findByPk(userId);
    if (!user) {
      return res.status(404).json({
        status: 'error',
        message: 'User not found'
      });
    }

    // Initialize preferences if they don't exist
    if (!user.preferences) {
      user.preferences = {};
    }

    // Update preferences
    if (notifications !== undefined) {
      user.preferences.notifications = notifications;
    }

    if (privacyLevel) {
      user.preferences.privacyLevel = privacyLevel;
    }

    // Save user
    await user.save();

    return res.status(200).json({
      status: 'success',
      message: 'Preferences updated successfully',
      data: {
        preferences: user.preferences
      }
    });
  } catch (error) {
    next(error);
  }
};

// Helper function to get user rank
const getUserRank = async (userId) => {
  try {
    // Get all users with their points
    const users = await models.User.findAll({
      attributes: ['id'],
      include: [
        {
          model: models.Submission,
          as: 'submissions',
          attributes: ['pointsAwarded'],
          where: {
            correct: true
          },
          required: false
        }
      ]
    });

    // Calculate total points for each user
    const userPoints = users.map(user => {
      const points = user.submissions.reduce((total, submission) => total + submission.pointsAwarded, 0);
      return {
        userId: user.id,
        points
      };
    });

    // Sort by points in descending order
    userPoints.sort((a, b) => b.points - a.points);

    // Find the index of the current user
    const userIndex = userPoints.findIndex(user => user.userId === userId);

    // Return the rank (index + 1)
    return userIndex !== -1 ? userIndex + 1 : userPoints.length;
  } catch (error) {
    logger.error('Error getting user rank:', error);
    return 0;
  }
};

// Helper function to calculate total user points
const calculateUserPoints = async (userId) => {
  try {
    // Sum up points from correct submissions
    const points = await models.Submission.sum('pointsAwarded', {
      where: {
        userId,
        correct: true
      }
    });

    // Sum up points from achievements
    const achievementPoints = await models.Achievement.sum('points', {
      where: {
        userId
      }
    });

    return (points || 0) + (achievementPoints || 0);
  } catch (error) {
    logger.error('Error calculating user points:', error);
    return 0;
  }
};