import { verifyToken } from '../utils/tokens.js';
import models from '../models/index.js';
import logger from '../utils/logger.js';

// Middleware to authenticate users
export const authenticate = async (req, res, next) => {
  try {
    // Get token from header
    const authHeader = req.headers.authorization;
    if (!authHeader || !authHeader.startsWith('Bearer ')) {
      return res.status(401).json({
        status: 'error',
        message: 'Authentication required. Please provide a valid token.'
      });
    }

    const token = authHeader.split(' ')[1];
    
    // Verify token
    const decoded = verifyToken(token);
    
    // Find user
    const user = await models.User.findByPk(decoded.userId);
    
    // Check if user exists or is banned
    if (!user || user.status === 'banned') {
      return res.status(401).json({
        status: 'error',
        message: 'Invalid or expired token'
      });
    }

    // Attach user to request
    req.user = user;
    next();
  } catch (error) {
    if (error.name === 'TokenExpiredError') {
      return res.status(401).json({
        status: 'error',
        message: 'Token expired',
        code: 'TOKEN_EXPIRED'
      });
    }
    
    logger.error('Authentication error:', error);
    return res.status(401).json({
      status: 'error',
      message: 'Authentication failed'
    });
  }
};

// Define role-based permissions
const permissions = {
  admin: {
    can_manage_users: true,
    can_create_ctf: true,
    can_manage_challenges: true,
    can_manage_settings: true,
    can_view_analytics: true,
    can_submit_writeups: true,
    can_access_premium_content: true,
    can_participate_ctf: true,
    can_view_leaderboard: true,
    can_join_teams: true
  },
  membershipUser: {
    can_manage_users: false,
    can_create_ctf: true,
    can_manage_challenges: true,
    can_manage_settings: false,
    can_view_analytics: false,
    can_submit_writeups: true,
    can_access_premium_content: true,
    can_participate_ctf: true,
    can_view_leaderboard: true,
    can_join_teams: true
  },
  user: {
    can_manage_users: false,
    can_create_ctf: false,
    can_manage_challenges: false,
    can_manage_settings: false,
    can_view_analytics: false,
    can_submit_writeups: true,
    can_access_premium_content: false,
    can_participate_ctf: true,
    can_view_leaderboard: true,
    can_join_teams: true
  }
};

// Middleware to authorize users based on permissions
export const authorize = (...requiredPermissions) => {
  return (req, res, next) => {
    try {
      if (!req.user) {
        return res.status(401).json({
          status: 'error',
          message: 'Authentication required'
        });
      }

      const userRole = req.user.role;
      const userPermissions = permissions[userRole];

      // Check if user has all required permissions
      const hasPermission = requiredPermissions.every(permission => 
        userPermissions && userPermissions[permission]
      );

      if (!hasPermission) {
        return res.status(403).json({
          status: 'error',
          message: 'You do not have permission to perform this action'
        });
      }

      next();
    } catch (error) {
      logger.error('Authorization error:', error);
      return res.status(500).json({
        status: 'error',
        message: 'Authorization error'
      });
    }
  };
};

// Middleware to ensure user can only access their own resources
export const ownershipCheck = (modelName, paramIdField = 'id', userIdField = 'userId') => {
  return async (req, res, next) => {
    try {
      // Skip for admins (they can access all resources)
      if (req.user.role === 'admin') {
        return next();
      }

      const resourceId = req.params[paramIdField];
      if (!resourceId) {
        return res.status(400).json({
          status: 'error',
          message: `Missing ${paramIdField} parameter`
        });
      }

      // Find the resource
      const resource = await models[modelName].findByPk(resourceId);
      
      if (!resource) {
        return res.status(404).json({
          status: 'error',
          message: 'Resource not found'
        });
      }

      // Check if the user owns the resource
      if (resource[userIdField] !== req.user.id) {
        return res.status(403).json({
          status: 'error',
          message: 'You do not have permission to access this resource'
        });
      }

      // Attach resource to request for future use
      req.resource = resource;
      next();
    } catch (error) {
      logger.error('Ownership check error:', error);
      return res.status(500).json({
        status: 'error',
        message: 'Ownership check error'
      });
    }
  };
};