import nodemailer from 'nodemailer';
import logger from './logger.js';

// Create reusable transporter
const createTransporter = () => {
  // In development, use ethereal email for testing
  if (process.env.NODE_ENV === 'development') {
    return nodemailer.createTransport({
      host: 'smtp.ethereal.email',
      port: 587,
      secure: false,
      auth: {
        user: 'ethereal.user@ethereal.email',
        pass: 'ethereal.password'
      }
    });
  }

  return nodemailer.createTransport({
    host: process.env.SMTP_HOST,
    port: process.env.SMTP_PORT,
    secure: process.env.SMTP_PORT === '465',
    auth: {
      user: process.env.SMTP_USER,
      pass: process.env.SMTP_PASSWORD
    }
  });
};

// Send email
export const sendEmail = async (options) => {
  try {
    // In development, just log the email content
    if (process.env.NODE_ENV === 'development') {
      logger.info('Email would be sent in production:', {
        to: options.to,
        subject: options.subject,
        html: options.html
      });
      return { messageId: 'development-mode' };
    }

    const transporter = createTransporter();
    
    const mailOptions = {
      from: `"${process.env.EMAIL_FROM_NAME || 'CTF Platform'}" <${process.env.EMAIL_FROM}>`,
      to: options.to,
      subject: options.subject,
      html: options.html
    };

    if (options.attachments) {
      mailOptions.attachments = options.attachments;
    }

    const info = await transporter.sendMail(mailOptions);
    logger.info(`Email sent: ${info.messageId}`);
    return info;
  } catch (error) {
    logger.error('Error sending email:', error);
    if (process.env.NODE_ENV === 'development') {
      // In development, don't throw the error
      return { messageId: 'development-mode-error' };
    }
    throw error;
  }
};

// Email templates
export const emailTemplates = {
  // Welcome email with verification link
  welcome: (name, verificationUrl) => ({
    subject: 'Welcome to CTF Platform - Please Verify Your Email',
    html: `
      <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
        <h2>Welcome to CTF Platform, ${name}!</h2>
        <p>Thank you for registering. To get started, please verify your email address by clicking the button below:</p>
        <div style="text-align: center; margin: 30px 0;">
          <a href="${verificationUrl}" style="background-color: #4CAF50; color: white; padding: 12px 20px; text-decoration: none; border-radius: 4px; font-weight: bold;">
            Verify Email Address
          </a>
        </div>
        <p>If the button doesn't work, you can also copy and paste this link into your browser:</p>
        <p>${verificationUrl}</p>
        <p>This link will expire in 24 hours.</p>
        <p>If you didn't create an account, you can safely ignore this email.</p>
        <hr style="margin: 30px 0; border: none; border-top: 1px solid #eee;">
        <p style="color: #666; font-size: 12px;">© ${new Date().getFullYear()} CTF Platform. All rights reserved.</p>
      </div>
    `
  }),
  
  // Password reset email
  passwordReset: (name, resetUrl) => ({
    subject: 'Reset Your CTF Platform Password',
    html: `
      <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
        <h2>Password Reset Request</h2>
        <p>Hi ${name},</p>
        <p>We received a request to reset your password. Click the button below to set a new password:</p>
        <div style="text-align: center; margin: 30px 0;">
          <a href="${resetUrl}" style="background-color: #2196F3; color: white; padding: 12px 20px; text-decoration: none; border-radius: 4px; font-weight: bold;">
            Reset Password
          </a>
        </div>
        <p>If the button doesn't work, you can also copy and paste this link into your browser:</p>
        <p>${resetUrl}</p>
        <p>This link will expire in 1 hour.</p>
        <p>If you didn't request a password reset, please ignore this email or contact support if you have concerns.</p>
        <hr style="margin: 30px 0; border: none; border-top: 1px solid #eee;">
        <p style="color: #666; font-size: 12px;">© ${new Date().getFullYear()} CTF Platform. All rights reserved.</p>
      </div>
    `
  }),
  
  // CTF approval notification
  ctfApproved: (name, ctfTitle, ctfUrl) => ({
    subject: 'Your CTF Has Been Approved',
    html: `
      <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
        <h2>Your CTF Has Been Approved!</h2>
        <p>Hi ${name},</p>
        <p>Great news! Your CTF "${ctfTitle}" has been reviewed and approved by our team.</p>
        <p>It is now visible to all users and ready for participation.</p>
        <div style="text-align: center; margin: 30px 0;">
          <a href="${ctfUrl}" style="background-color: #FF5722; color: white; padding: 12px 20px; text-decoration: none; border-radius: 4px; font-weight: bold;">
            View Your CTF
          </a>
        </div>
        <p>Thank you for contributing to our platform!</p>
        <hr style="margin: 30px 0; border: none; border-top: 1px solid #eee;">
        <p style="color: #666; font-size: 12px;">© ${new Date().getFullYear()} CTF Platform. All rights reserved.</p>
      </div>
    `
  })
};